import type { FC } from 'react';
import { Fragment } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { groupBy } from 'tachyon-utils';
import { Layout } from 'twitch-core-ui';
import { VideoCard } from '../../common';
import { ScheduleTitle } from './ScheduleTitle';
import type { PastBroadcasts_items } from './__generated__/PastBroadcasts_items.graphql';
import { olderThanOneWeek } from './utils';

type PastBroadcastsProps = {
  items: PastBroadcasts_items;
};

// istanbul ignore next: trivial
export const PastBroadcastsBase: FC<PastBroadcastsProps> = ({ items }) => {
  const { formatDate } = useIntl();

  items = items
    .filter((item) => !!item.publishedAt && !olderThanOneWeek(item.publishedAt))
    .sort(
      (a, b) =>
        new Date(a.publishedAt ?? 0).getTime() -
        new Date(b.publishedAt ?? 0).getTime(),
    );

  const segments = groupBy(items, (item) =>
    formatDate(new Date(item.publishedAt as string), {
      day: 'numeric',
      month: 'long',
      weekday: 'long',
    }),
  );

  return (
    <>
      {Object.entries(segments)
        .reverse()
        .map(([title, videos]) => (
          <Fragment key={title}>
            <Layout padding={{ bottom: 2 }}>
              <ScheduleTitle>{title}</ScheduleTitle>
            </Layout>
            {videos.map((video) => (
              <VideoCard
                key={video.id}
                tagPage={TagPage.Stream}
                video={video}
              />
            ))}
          </Fragment>
        ))}
    </>
  );
};

PastBroadcastsBase.displayName = 'PastBroadcasts';

export const PastBroadcasts = createFragmentContainer(PastBroadcastsBase, {
  items: graphql`
    fragment PastBroadcasts_items on Video @relay(plural: true) {
      id
      publishedAt
      ...VideoCard_video
    }
  `,
});
