import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { isAndroidOS, isIOS, useStaticEnvironment } from 'tachyon-environment';
import { useIntl } from 'tachyon-intl';
import { formatTimeRange, getBoxArtSrcAndSrcSet } from 'tachyon-more-ui';
import {
  AspectRatio,
  CardImage,
  Color,
  CoreText,
  Display,
  FontSize,
  Layout,
  TextType,
} from 'twitch-core-ui';
import { RemindMeButton } from './RemindMeButton';
import type { ScheduleSegmentCard_segment } from './__generated__/ScheduleSegmentCard_segment.graphql';
import type { TimeRange } from './utils';

type ScheduleSegmentCardProps = {
  interruption: TimeRange | null;
  segment: ScheduleSegmentCard_segment;
};

// istanbul ignore next: trivial
const ScMetaContainer = styled(Layout)<{ $interrupted: boolean }>`
  display: flex;
  flex-direction: column;
  justify-content: space-evenly;
  ${(p) => (p.$interrupted ? `text-decoration: line-through` : '')};
`;

// istanbul ignore next: trivial
export const ScheduleSegmentCardBase: FC<ScheduleSegmentCardProps> = ({
  interruption,
  segment,
}) => {
  const intl = useIntl();
  const game = segment.categories?.[0];
  const segmentStart = new Date(segment.startAt);
  const segmentEnd = segment.endAt ? new Date(segment.endAt) : null;

  const interrupted =
    !!interruption &&
    segmentStart >= interruption.startAt &&
    segmentStart < interruption.endAt;

  const timeLabel = formatTimeRange(intl, segmentStart, segmentEnd);

  const { client } = useStaticEnvironment();
  const isAndroidOrIos =
    isAndroidOS(client.agentInfo) || isIOS(client.agentInfo);

  return (
    <Layout as="article" display={Display.Flex} margin={{ bottom: 2 }}>
      <ScMetaContainer
        $interrupted={interrupted || segment.isCancelled}
        flexOrder={2}
      >
        {game?.displayName && (
          <CoreText
            color={Color.Base}
            fontSize={FontSize.Size5}
            type={TextType.H3}
          >
            {game.displayName}
          </CoreText>
        )}
        {segment.title && (
          <CoreText color={Color.Alt} fontSize={FontSize.Size6}>
            {segment.title}
          </CoreText>
        )}
        <CoreText
          color={Color.Alt}
          data-test-selector="time-label"
          fontSize={FontSize.Size6}
        >
          {timeLabel}
        </CoreText>
        {isAndroidOrIos && <RemindMeButton />}
      </ScMetaContainer>
      <Layout
        flexOrder={1}
        flexShrink={0}
        margin={{ right: 1 }}
        style={{ width: '6.4rem' }}
      >
        <CardImage
          alt=""
          aspect={AspectRatio.BoxArt}
          {...getBoxArtSrcAndSrcSet(game?.boxArtURL)}
        />
      </Layout>
    </Layout>
  );
};

ScheduleSegmentCardBase.displayName = 'ScheduleSegmentCard';

export const ScheduleSegmentCard = createFragmentContainer(
  ScheduleSegmentCardBase,
  {
    segment: graphql`
      fragment ScheduleSegmentCard_segment on ScheduleSegment {
        id
        title
        startAt
        endAt
        isCancelled
        categories {
          id
          displayName
          boxArtURL(width: 144, height: 192)
        }
      }
    `,
  },
);
