import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { Layout } from 'twitch-core-ui';
import {
  ChannelFeaturedCardStream,
  ChannelFeaturedCardVideo,
} from '../../common';
import { ScheduleSegmentCard } from './ScheduleSegmentCard';
import { ScheduleTitle } from './ScheduleTitle';
import type { TodaySection_interruption } from './__generated__/TodaySection_interruption.graphql';
import type { TodaySection_segments } from './__generated__/TodaySection_segments.graphql';
import type { TodaySection_stream } from './__generated__/TodaySection_stream.graphql';
import type { TodaySection_videos } from './__generated__/TodaySection_videos.graphql';
import { getTimeRange } from './utils';

type TodaySectionProps = {
  interruption: TodaySection_interruption | null;
  segments: TodaySection_segments;
  stream: TodaySection_stream | null;
  videos: TodaySection_videos;
};

// istanbul ignore next: trivial
export const TodaySectionBase: FC<TodaySectionProps> = ({
  interruption,
  segments,
  stream,
  videos,
}) => {
  const { formatMessage } = useIntl();

  return (
    <>
      <Layout padding={{ bottom: 2 }}>
        <ScheduleTitle>
          {formatMessage('Today', 'ChannelSchedulePageTodaySection')}
        </ScheduleTitle>
      </Layout>
      {videos.map((video) => (
        <ChannelFeaturedCardVideo
          key={video.id}
          type="past_broadcast"
          video={video}
        />
      ))}
      {stream && <ChannelFeaturedCardStream stream={stream} type="live" />}
      {segments.map((segment) => (
        <ScheduleSegmentCard
          interruption={getTimeRange(interruption)}
          key={segment.id}
          segment={segment}
        />
      ))}
    </>
  );
};

TodaySectionBase.displayName = 'TodaySection';

export const TodaySection = createFragmentContainer(TodaySectionBase, {
  interruption: graphql`
    fragment TodaySection_interruption on ScheduleInterruption {
      startAt
      endAt
    }
  `,
  segments: graphql`
    fragment TodaySection_segments on ScheduleSegment @relay(plural: true) {
      id
      startAt
      ...ScheduleSegmentCard_segment
    }
  `,
  stream: graphql`
    fragment TodaySection_stream on Stream {
      id
      ...ChannelFeaturedCardStream_stream
    }
  `,
  videos: graphql`
    fragment TodaySection_videos on Video @relay(plural: true) {
      id
      ...ChannelFeaturedCardVideo_video
    }
  `,
});
