import { graphql } from 'react-relay/hooks';
import { useIntl } from 'tachyon-intl';
import { ClientOnly } from 'tachyon-more-ui';
import type { ChannelPathParameters } from 'tachyon-page-utils';
import {
  channelPathGetInitialProps,
  channelSchedulePageviewTracking,
} from 'tachyon-page-utils';
import { partition, reduceToNonNullNodes } from 'tachyon-utils';
import { Layout, SVGAsset } from 'twitch-core-ui';
import { ChannelEmptyState } from '../../common';
import { pageHeadQueryVariables } from '../../growth';
import { ChannelLayout } from '../../layouts';
import type { PageWithQueryProps, TomorrowPage } from '../types';
import { FutureSegments } from './FutureSegments';
import { LastStreamTitle } from './LastStreamTitle';
import { LiveTitle } from './LiveTitle';
import { NextSegmentTitle } from './NextSegmentTitle';
import { PastBroadcasts } from './PastBroadcasts';
import { TodaySection } from './TodaySection';
import type {
  ChannelSchedulePage_QueryResponse,
  ChannelSchedulePage_QueryVariables,
} from './__generated__/ChannelSchedulePage_Query.graphql';
import { isToday } from './utils';

type ChannelSchedulePageInitialProps = {
  queryVariables: ChannelSchedulePage_QueryVariables;
};

type ChannelSchedulePageProps = PageWithQueryProps<
  ChannelSchedulePage_QueryResponse,
  ChannelSchedulePageInitialProps
>;

export const ChannelSchedulePage: TomorrowPage<
  ChannelSchedulePageInitialProps,
  ChannelSchedulePageProps,
  ChannelPathParameters
> = ({ channel, loading, queryVariables }) => {
  const { formatMessage } = useIntl();

  let broadcasts = reduceToNonNullNodes(channel?.pastBroadcasts?.edges);
  if (channel?.stream) {
    broadcasts = broadcasts.slice(1);
  }

  const [todaysBroadcasts, pastBroadcasts] = partition(
    broadcasts,
    (item) => !!item.publishedAt && isToday(new Date(item.publishedAt)),
  );

  const segments = reduceToNonNullNodes(
    channel?.channel?.schedule?.segmentList?.edges,
  );
  const [todaysSegments, futureSegments] = partition(segments, (segment) =>
    isToday(new Date(segment.startAt)),
  );

  const scheduleInterruption = channel?.channel?.schedule?.interruption ?? null;
  const nextSegment = channel?.channel?.schedule?.nextSegment;

  let title: JSX.Element | null = null;
  if (channel) {
    if (channel.stream) {
      title = <LiveTitle user={channel} />;
    } else if (nextSegment && !nextSegment.isCancelled) {
      title = <NextSegmentTitle segment={nextSegment} user={channel} />;
    } else if (pastBroadcasts.length > 0) {
      title = <LastStreamTitle user={channel} video={pastBroadcasts[0]} />;
    }
  }

  const showTodayContent =
    !!channel?.stream ||
    todaysSegments.length > 0 ||
    todaysBroadcasts.length > 0;

  const showContent =
    pastBroadcasts.length > 0 ||
    !!title ||
    showTodayContent ||
    futureSegments.length > 0;

  return (
    <ChannelLayout
      loading={loading}
      login={queryVariables.login}
      user={channel}
    >
      <ClientOnly>
        {channel &&
          (!showContent ? (
            <ChannelEmptyState
              asset={SVGAsset.Events}
              title={formatMessage(
                'This channel currently has no schedule',
                'ChannelSchedulePage',
              )}
            />
          ) : (
            <>
              {pastBroadcasts.length > 0 && (
                <PastBroadcasts items={pastBroadcasts} />
              )}
              {title && (
                <div
                  ref={(ref) =>
                    // TODO: Safari polyfill
                    // https://caniuse.com/mdn-api_element_scrollintoview_scrollintoviewoptions
                    ref?.scrollIntoView({ behavior: 'smooth' })
                  }
                >
                  <Layout padding={{ bottom: 2, top: 1 }}>{title}</Layout>
                </div>
              )}
              {showTodayContent && (
                <TodaySection
                  interruption={scheduleInterruption}
                  segments={todaysSegments}
                  stream={channel.stream}
                  videos={todaysBroadcasts}
                />
              )}
              {futureSegments.length > 0 && (
                <FutureSegments
                  interruption={scheduleInterruption}
                  segments={futureSegments}
                />
              )}
            </>
          ))}
      </ClientOnly>
    </ChannelLayout>
  );
};

ChannelSchedulePage.displayName = 'ChannelSchedulePage';
ChannelSchedulePage.pageviewTracking = channelSchedulePageviewTracking;
// TODO: verify non-js interactivity after dev
ChannelSchedulePage.requiresJsForInteractivity = true;
ChannelSchedulePage.handlesLoading = true;
ChannelSchedulePage.getInitialProps = (ctx) =>
  channelPathGetInitialProps(ctx, pageHeadQueryVariables(ctx));

ChannelSchedulePage.query = graphql`
  query ChannelSchedulePage_Query($login: String!, $url: String!) {
    channel: user(login: $login) {
      ...ChannelLayout_user
      id
      displayName
      login
      pastBroadcasts: videos(first: 30, type: ARCHIVE, sort: TIME) {
        edges {
          node {
            id
            publishedAt
            ...PastBroadcasts_items
            ...TodaySection_videos
            ...LastStreamTitle_video
          }
        }
      }
      stream {
        id
        ...TodaySection_stream
      }
      channel {
        schedule {
          id
          interruption {
            ...TodaySection_interruption
            ...FutureSegments_interruption
          }

          segmentList(first: 30) {
            edges {
              node {
                startAt
                ...TodaySection_segments
                ...FutureSegments_segments
              }
            }
          }

          nextSegment {
            id
            isCancelled
            ...NextSegmentTitle_segment
          }
        }
      }
    }
    ...PageHead_query
  }
`;
