import { datatype, internet, lorem } from 'faker';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { ChannelEmptyState } from '../../common';
import { FutureSegments } from './FutureSegments';
import { LastStreamTitle } from './LastStreamTitle';
import { LiveTitle } from './LiveTitle';
import { NextSegmentTitle } from './NextSegmentTitle';
import { PastBroadcasts } from './PastBroadcasts';
import { TodaySection } from './TodaySection';
import { getRelativeDate } from './utils';
import { ChannelSchedulePage } from '.';

describe(ChannelSchedulePage, () => {
  const setup = createShallowWrapperFactory(ChannelSchedulePage, () => {
    const login = lorem.word().toLowerCase();

    return {
      ' $fragmentRefs': {
        PageHead_query: true,
      },
      channel: {
        ' $fragmentRefs': {
          ChannelLayout_user: true,
        },
        channel: {
          schedule: {
            id: datatype.uuid(),
            interruption: null,
            nextSegment: null,
            segmentList: null,
          },
        },
        displayName: 'Arthas',
        id: datatype.uuid(),
        login,
        pastBroadcasts: null,
        stream: null,
      },
      loading: false,
      queryVariables: {
        login,
        url: internet.url(),
      },
    };
  });

  const today = new Date();

  it('renders empty state', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ChannelEmptyState)).toExist();
  });

  describe('when there are only past broadcasts', () => {
    it('renders past broadcasts', () => {
      const { wrapper } = setup({
        channel: {
          pastBroadcasts: {
            edges: [
              {
                node: {
                  id: datatype.uuid(),
                  publishedAt: '',
                },
              },
            ],
          },
        },
      });

      expect(wrapper.find(PastBroadcasts)).toExist();
    });

    it('renders past broadcast title', () => {
      const { wrapper } = setup({
        channel: {
          pastBroadcasts: {
            edges: [
              {
                node: {
                  id: datatype.uuid(),
                  publishedAt: getRelativeDate(today, {
                    days: -5,
                  }).toISOString(),
                },
              },
            ],
          },
        },
      });

      expect(wrapper.find(LastStreamTitle)).toExist();
    });

    it("renders today's past broadcasts", () => {
      const { wrapper } = setup({
        channel: {
          pastBroadcasts: {
            edges: [
              {
                node: {
                  id: datatype.uuid(),
                  publishedAt: today.toISOString(),
                },
              },
            ],
          },
        },
      });

      expect(wrapper.find(TodaySection).prop('videos')).toHaveLength(1);
    });
  });

  describe('when there is next segment', () => {
    it('renders next segment title', () => {
      const { wrapper } = setup({
        channel: {
          channel: {
            schedule: {
              nextSegment: {
                ' $fragmentRefs': { NextSegmentTitle_segment: true },
                id: datatype.uuid(),
                isCancelled: false,
              },
            },
          },
        },
      });

      expect(wrapper.find(NextSegmentTitle)).toExist();
    });
  });

  describe('when channel is live', () => {
    it('renders live title', () => {
      const { wrapper } = setup({
        channel: {
          stream: {
            ' $fragmentRefs': { TodaySection_stream: true },
            id: datatype.uuid(),
          },
        },
      });

      expect(wrapper.find(LiveTitle)).toExist();
    });
  });

  describe('when there are schedule segments', () => {
    it('renders future segments', () => {
      const { wrapper } = setup({
        channel: {
          channel: {
            schedule: {
              segmentList: {
                edges: [
                  {
                    node: {
                      ' $fragmentRefs': {
                        FutureSegments_segments: true,
                        TodaySection_segments: true,
                      },
                      startAt: getRelativeDate(today, {
                        days: 5,
                      }).toISOString(),
                    },
                  },
                ],
              },
            },
          },
        },
      });

      expect(wrapper.find(FutureSegments)).toExist();
    });

    it("renders today's segments", () => {
      const { wrapper } = setup({
        channel: {
          channel: {
            schedule: {
              segmentList: {
                edges: [
                  {
                    node: {
                      ' $fragmentRefs': {
                        FutureSegments_segments: true,
                        TodaySection_segments: true,
                      },
                      startAt: today.toISOString(),
                    },
                  },
                ],
              },
            },
          },
        },
      });

      expect(wrapper.find(TodaySection).prop('segments')).toHaveLength(1);
    });
  });
});
