type SerializedTimeRange = { endAt: string; startAt: string };
type SerializedEndlessTimeRange = { endAt: null; startAt: string };

export type TimeRange = { endAt: Date; startAt: Date };
type EndlessTimeRange = { endAt: null; startAt: Date };

export function getTimeRange(
  serialized: SerializedTimeRange | null,
): TimeRange | null;
export function getTimeRange(
  serialized: SerializedEndlessTimeRange | null,
): EndlessTimeRange | null;

export function getTimeRange(
  serialized: SerializedEndlessTimeRange | SerializedTimeRange | null,
): EndlessTimeRange | TimeRange | null {
  if (!serialized) {
    return null;
  }

  const startAt = new Date(serialized.startAt);

  if (serialized.endAt) {
    return {
      endAt: new Date(serialized.endAt),
      startAt,
    };
  }

  return {
    endAt: null,
    startAt: new Date(serialized.startAt),
  };
}

export const isToday = (date: Date): boolean => {
  const today = new Date();
  return today.toDateString() === date.toDateString();
};

const ONE_WEEK_IN_MS = 7 * 24 * 60 * 60 * 1000;

export const olderThanOneWeek = (dateStr: string): boolean => {
  const date = new Date(dateStr).getTime();
  const today = Date.now();

  return today - date > ONE_WEEK_IN_MS;
};

export const getRelativeDate = (
  date: Date,
  offset: { days?: number },
): Date => {
  const newDate = new Date(date.getTime());
  if (offset.days) {
    newDate.setDate(date.getDate() + offset.days);
  }

  return newDate;
};
