import type { FC } from 'react';
import { useEffect } from 'react';
import type { RelayRefetchProp } from 'react-relay/legacy';
import { createRefetchContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { formatLiveViewersCount } from 'tachyon-more-ui';
import {
  AlignItems,
  AlignSelf,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Layout,
  LineHeight,
  Overflow,
  SVG,
  SVGAsset,
  SVGType,
  WhiteSpace,
} from 'twitch-core-ui';
import { useTheme } from '../../../../theme';
import { ContentInfoBoxContainer, InfoBoxAvatar } from '../../../common';
import type { StandaloneStreamInfoBox_channel } from './__generated__/StandaloneStreamInfoBox_channel.graphql';

const ScMaxWidthLayout = styled(Layout)`
  max-width: 100%;
`;

const REFETCH_INTERVAL_MS = 60000;

export interface StandaloneStreamInfoBoxProps {
  channel: StandaloneStreamInfoBox_channel;
  relay: RelayRefetchProp;
}

export const StandaloneStreamInfoBoxBase: FC<StandaloneStreamInfoBoxProps> = ({
  channel,
  relay,
}) => {
  const { formatMessage } = useIntl();
  const { currentTheme } = useTheme();
  const { broadcastSettings, displayName, login, stream } = channel;

  useEffect(() => {
    const interval = setInterval(() => {
      relay.refetch({ login }, null, null, { force: true });
    }, REFETCH_INTERVAL_MS);
    return () => {
      clearInterval(interval);
    };
  }, [login, relay]);

  return (
    <ContentInfoBoxContainer>
      <ScMaxWidthLayout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        flexGrow={1}
        flexShrink={0}
      >
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexDirection={FlexDirection.Row}
        >
          <Layout
            alignSelf={AlignSelf.Center}
            display={Display.Flex}
            padding={1}
          >
            <InfoBoxAvatar owner={channel} withoutLink />
          </Layout>
          <Layout
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            flexGrow={1}
            overflow={Overflow.Auto}
          >
            <CoreText
              color={Color.Base}
              fontSize={FontSize.Size3}
              lineHeight={LineHeight.Heading}
              whiteSpace={WhiteSpace.NoWrap}
            >
              {displayName || login}
            </CoreText>
            <Layout overflow={Overflow.Auto}>
              {broadcastSettings?.title && (
                <CoreText
                  color={Color.Alt}
                  fontSize={FontSize.Size6}
                  lines={2}
                  whiteSpace={WhiteSpace.PreWrap}
                >
                  {broadcastSettings.title}
                </CoreText>
              )}
            </Layout>
            <Layout overflow={Overflow.Auto}>
              {stream?.viewersCount && (
                <CoreText
                  color={Color.Alt}
                  fontSize={FontSize.Size6}
                  whiteSpace={WhiteSpace.NoWrap}
                >
                  {formatLiveViewersCount(formatMessage, stream.viewersCount)}
                </CoreText>
              )}
            </Layout>
          </Layout>
          <Layout padding={{ x: 2 }}>
            <SVG
              asset={SVGAsset.LogoTwitchWordmark}
              height={70}
              type={currentTheme === 'dark' ? SVGType.Alt : SVGType.Brand}
              width={70}
            />
          </Layout>
        </Layout>
      </ScMaxWidthLayout>
    </ContentInfoBoxContainer>
  );
};

StandaloneStreamInfoBoxBase.displayName = 'StandaloneStreamInfoBox';

export const StandaloneStreamInfoBox = createRefetchContainer(
  StandaloneStreamInfoBoxBase,
  {
    channel: graphql`
      fragment StandaloneStreamInfoBox_channel on User {
        login
        displayName
        profileImageURL(width: 50)
        broadcastSettings {
          title
        }
        stream {
          viewersCount
        }
      }
    `,
  },
  graphql`
    query StandaloneStreamInfoBoxChannelRefetchQuery($login: String!) {
      channel: user(login: $login) {
        ...StandaloneStreamInfoBox_channel
      }
    }
  `,
);
