import { useRef } from 'react';
import { graphql } from 'react-relay/hooks';
import type { ChatWorkerInstance } from 'tachyon-chat';
import { ChatRoot } from 'tachyon-chat-ui';
import type {
  ChannelInitialProps,
  ChannelPathParameters,
} from 'tachyon-page-utils';
import {
  channelIsFound,
  channelPathGetInitialProps,
  channelWatchPageviewTracking,
} from 'tachyon-page-utils';
import { convertToUnsafeID } from 'tachyon-relay';
import { getFormattedUserDisplayName, useEffectOnce } from 'tachyon-utils';
import { Background, Display, FlexDirection, Layout } from 'twitch-core-ui';
import { CLIENT_ID } from '../../../config';
// @ts-expect-error: WebWorkers are necessarily (untyped) JS
// eslint-disable-next-line import/no-internal-modules
import ChatWorker from '../../../workers/chat.worker';
import { PulsarStreamPlayer } from '../../Player';
import { ChatPane } from '../../common';
import { NotFoundErrorLayout, Page, PlayerLayout } from '../../layouts';
import type { TomorrowPage } from '../types';
import { SafetyDirector } from './SafetyDirector';
import { StandaloneStreamInfoBox } from './StandaloneStreamInfoBox';
import type { ChannelStandalone_QueryResponse } from './__generated__/ChannelStandalone_Query.graphql';

// Page can be accessed at the following path:
// /monstercat/kntkhbcmmyvxkmnuczri?sxfgohjs=pzxonfvp

/**
 * The user is able to scroll back through chat history. Set this sufficiently
 * high that they get some history, but this doesn't become a performance concern
 * since the messages aren't virtualized.
 */
const CHAT_BUFFER_SIZE = 100;

// Timing threshold within which we consider 2 taps a single double-tap event
const DOUBLE_TAP_WINDOW_MS = 300;

export type ChannelStandaloneProps = ChannelInitialProps &
  ChannelStandalone_QueryResponse;

export const ChannelStandalone: TomorrowPage<
  ChannelInitialProps,
  ChannelStandaloneProps,
  ChannelPathParameters
> = ({ channel }) => {
  const lastTouchEndRef = useRef(0);

  useEffectOnce(() => {
    const doubleTapPreventionHandler = ({ preventDefault }: TouchEvent) => {
      const now = new Date().valueOf();
      if (now - lastTouchEndRef.current <= DOUBLE_TAP_WINDOW_MS) {
        preventDefault();
      }

      lastTouchEndRef.current = now;
    };

    document.addEventListener('touchend', doubleTapPreventionHandler);

    return () => {
      document.removeEventListener('touchend', doubleTapPreventionHandler);
    };
  });

  if (!channelIsFound(channel)) {
    return <NotFoundErrorLayout />;
  }

  const { hosting, login } = channel;
  const displayName = getFormattedUserDisplayName(channel);

  // ChannelStandalone is an embedded experience, thus we want to hide the
  // top nav that would otherwise allow you to navigate off the page.
  return (
    <>
      <SafetyDirector login={login} />
      <Page hideTopNav>
        <PlayerLayout player={<PulsarStreamPlayer channel={channel} />}>
          <Layout
            background={Background.Base}
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            fullHeight
          >
            <StandaloneStreamInfoBox channel={channel} />
            <ChatRoot
              bufferSize={CHAT_BUFFER_SIZE}
              channel={{ id: convertToUnsafeID(channel.id), login }}
              clientApiId={CLIENT_ID}
              getWorker={() =>
                // eslint-disable-next-line @typescript-eslint/no-unsafe-call
                new ChatWorker() as ChatWorkerInstance
              }
            >
              {!hosting && (
                <ChatPane channelDisplayName={displayName} preventInputUpsell />
              )}
            </ChatRoot>
          </Layout>
        </PlayerLayout>
      </Page>
    </>
  );
};

ChannelStandalone.displayName = 'ChannelStandalonePage';
ChannelStandalone.getInitialProps = channelPathGetInitialProps;
ChannelStandalone.pageviewTracking = channelWatchPageviewTracking;
ChannelStandalone.staleWhileRevalidate = 30;
ChannelStandalone.totalCacheLife = 60;
ChannelStandalone.requiresJsForInteractivity = true;

ChannelStandalone.query = graphql`
  query ChannelStandalone_Query($login: String!) {
    channel: user(login: $login) {
      ...StreamPlayer_channel @arguments(standalone: true)
      ...StandaloneStreamInfoBox_channel
      id
      login
      displayName
      stream {
        id
      }
      hosting {
        id
      }
    }
  }
`;
