import type { FC } from 'react';
import type { RelayRefetchProp } from 'react-relay/legacy';
import { createRefetchContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import {
  CHANNEL_VODS_FILTER_QUERY_PARAM_KEY,
  getChannelVodsFilter,
} from 'tachyon-page-utils';
import { useRefetchList } from 'tachyon-relay';
import {
  AlignItems,
  Color,
  Column,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  FontWeight,
  Grid,
  JustifyContent,
  Layout,
  Overflow,
  ResponsiveWrapper,
  SVGAsset,
  TextTransform,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { ChannelEmptyState, InfiniteList, VideoCard } from '../../../common';
import type {
  ChannelVideosShelves_channel,
  VideoShelfType,
} from './__generated__/ChannelVideosShelves_channel.graphql';
import type { ChannelVideosShelves_videoItem as VideoItem } from './__generated__/ChannelVideosShelves_videoItem.graphql';

export type ChannelVideosShelvesProps = {
  activeShelfType: VideoShelfType | null;
  channel: ChannelVideosShelves_channel;
  relay: RelayRefetchProp;
};

const LOAD_MORE_LIMIT = 10;

const ScVideoCardContainer = styled(Layout)`
  width: 24rem;
`;

type VideoNodes = {
  description: string | null;
  id: string;
  items: VideoItem[];
  title: string;
  type: VideoShelfType;
};

export const ChannelVideosShelvesBase: FC<ChannelVideosShelvesProps> = ({
  activeShelfType,
  channel,
  relay,
}) => {
  const { formatMessage } = useIntl();

  const { endCursor, noMore, nodes } = useRefetchList(
    channel.videoShelves?.edges,
  );

  // require that all shelves only contain videos
  const videoShelves: VideoNodes[] = nodes
    .map((shelf) => {
      // only show video shelves on video page. clips shelves will be shown
      // on the clips page
      const shelfItems: VideoItem[] = (shelf.items ?? []).filter(
        (item): item is VideoItem => item.__typename === 'Video',
      );
      return { ...shelf, items: shelfItems };
    })
    .filter((shelf) => shelf.items.length > 0);

  const shelfRenderer = (idx: number): JSX.Element => {
    const shelf = videoShelves[idx];
    const filter = getChannelVodsFilter(shelf.type);
    return (
      <Layout key={shelf.id} margin={{ bottom: 1 }}>
        <Layout
          display={Display.Flex}
          fontSize={FontSize.Size5}
          fontWeight={FontWeight.SemiBold}
          justifyContent={JustifyContent.Between}
          margin={{ bottom: 0.5 }}
        >
          <CoreText
            color={Color.Alt2}
            fontWeight={FontWeight.SemiBold}
            transform={TextTransform.Uppercase}
          >
            {shelf.title}
          </CoreText>
          {filter && (
            <CoreLink
              linkTo="/deferToRenderLink"
              renderLink={renderTachyonLink({
                query: {
                  [CHANNEL_VODS_FILTER_QUERY_PARAM_KEY]: filter,
                },
                route: RouteName.ChannelVideos,
                routeParams: { login: channel.login },
              })}
            >
              <CoreText color={Color.Link} transform={TextTransform.Uppercase}>
                {formatMessage('View All', 'ChannelVideosShelf')}
              </CoreText>
            </CoreLink>
          )}
        </Layout>
        <Layout display={Display.Flex} overflow={Overflow.Scroll}>
          {shelf.items.map((item) => {
            return (
              <ScVideoCardContainer
                flexShrink={0}
                key={item.id}
                margin={{ x: 0.5 }}
              >
                <VideoCard tagPage={TagPage.Stream} video={item} />
              </ScVideoCardContainer>
            );
          })}
        </Layout>
      </Layout>
    );
  };

  const loadMoreShelves = () => {
    if (!noMore) {
      relay.refetch({ cursor: endCursor, first: 10, login: channel.login });
    }
  };

  const activeShelf = videoShelves.find(
    (shelf) => shelf.type === activeShelfType,
  );

  if (!activeShelf) {
    return videoShelves.length === 0 ? (
      <ChannelEmptyState
        asset={SVGAsset.MediaVideo}
        message={formatMessage(
          'It’s quiet... too quiet...',
          'ChannelVideosShelf',
        )}
        title={formatMessage(
          'This channel has no videos',
          'ChannelVideosShelf',
        )}
      />
    ) : (
      <InfiniteList
        itemRenderer={shelfRenderer}
        length={videoShelves.length}
        loadMore={loadMoreShelves}
        pageSize={LOAD_MORE_LIMIT}
      />
    );
  }

  return (
    <Layout
      alignItems={AlignItems.Center}
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
    >
      <Layout margin={{ y: 1 }}>
        <CoreText
          color={Color.Alt2}
          fontWeight={FontWeight.SemiBold}
          transform={TextTransform.Uppercase}
        >
          {activeShelf.title}
        </CoreText>
      </Layout>
      <ResponsiveWrapper centered>
        <Grid justifyContent={JustifyContent.Center}>
          {activeShelf.items.map((item) => (
            <Column
              cols={{ default: 12, lg: 4, md: 6, sm: 6, xl: 4, xs: 12, xxl: 3 }}
              key={item.id}
            >
              <VideoCard tagPage={TagPage.Stream} video={item} />
            </Column>
          ))}
        </Grid>
      </ResponsiveWrapper>
    </Layout>
  );
};

ChannelVideosShelvesBase.displayName = 'ChannelVideosShelves';

export const ChannelVideosShelves = createRefetchContainer(
  ChannelVideosShelvesBase,
  {
    channel: graphql`
      fragment ChannelVideosShelves_channel on User {
        login
        videoShelves(first: $first, after: $cursor) {
          edges {
            cursor
            node {
              id
              title
              description
              type
              items {
                ...ChannelVideosShelves_videoItem @relay(mask: false)
              }
            }
          }
          pageInfo {
            hasNextPage
          }
        }
      }
    `,
  },
  graphql`
    query ChannelVideosShelves_RefetchQuery(
      $login: String!
      $first: Int!
      $cursor: Cursor
    ) {
      channel: user(login: $login) {
        login
        ...ChannelVideosShelves_channel
      }
    }
  `,
);

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ChannelVideosShelves_videoItem on Video {
    id
    __typename
    ...VideoCard_video
  }
`;
