import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { useConst } from 'tachyon-utils';
import {
  ButtonSize,
  Display,
  FlexDirection,
  Layout,
  Overflow,
} from 'twitch-core-ui';
import {
  ContentInfoBox,
  WEB_SHARE_BUTTON_HEIGHT,
  WebShareButton,
  canShare,
} from '../../../common';
import { ClipList } from '../ClipList';
import type { ClipPanel_clip } from './__generated__/ClipPanel_clip.graphql';

export type ClipPanelProps = {
  clip: ClipPanel_clip;
};

export const ClipPanelBase: FC<ClipPanelProps> = ({ clip }) => {
  const sharingSupported = useConst(canShare);
  const { formatMessage } = useIntl();
  const { broadcaster } = clip;

  return (
    <Layout
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      fullHeight
      overflow={Overflow.Hidden}
    >
      <ContentInfoBox
        contentMeta={{
          description: clip.title,
          header: broadcaster?.displayName ?? null,
        }}
        owner={
          broadcaster
            ? {
                displayName: broadcaster.displayName,
                login: broadcaster.login,
                profileImageURL: broadcaster.profileImageURL,
              }
            : null
        }
      />
      {sharingSupported && (
        <Layout height={WEB_SHARE_BUTTON_HEIGHT} padding={1}>
          <WebShareButton fullWidth size={ButtonSize.Large}>
            {formatMessage('Share this Clip', 'DiscoverMore')}
          </WebShareButton>
        </Layout>
      )}
      {clip.broadcaster?.clips && <ClipList clips={clip.broadcaster.clips} />}
    </Layout>
  );
};

export const ClipPanel = createFragmentContainer(ClipPanelBase, {
  clip: graphql`
    fragment ClipPanel_clip on Clip {
      broadcaster {
        clips(criteria: { period: LAST_WEEK, sort: VIEWS_DESC }) {
          ...ClipList_clips
        }
        login
        displayName
        profileImageURL(width: 50)
      }
      title
    }
  `,
});
