import type { FC } from 'react';
import { useCallback } from 'react';
import type { RelayRefetchProp } from 'react-relay/legacy';
import { createRefetchContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { useRefetchList } from 'tachyon-relay';
import { Layout } from 'twitch-core-ui';
import { CHANNELS_PAGE_SIZE } from '../../../../config';
import { ActiveTagFilter } from '../../../common';
import {
  ChannelList,
  GameCardCommon,
  NoChannelsLive,
} from '../../../discovery';
import type { TopChannelsForGame_query } from './__generated__/TopChannelsForGame_query.graphql';

export type TopChannelsForGameProps = {
  /**
   * Used for fallback when game is unknown
   */
  gameAlias: string;
  query: TopChannelsForGame_query;
  relay: RelayRefetchProp;
  tags: readonly string[] | null | undefined;
};

export const TopChannelsForGameBase: FC<TopChannelsForGameProps> = ({
  gameAlias,
  query,
  relay,
  tags,
}) => {
  const { game } = query;
  const {
    endCursor,
    noMore,
    nodes: streams,
  } = useRefetchList(query.game?.streams?.edges);

  const loadMore = useCallback(() => {
    if (noMore) {
      return;
    }

    relay.refetch(() => ({
      channelsCount: CHANNELS_PAGE_SIZE * 3,
      cursor: endCursor,
    }));
  }, [relay, endCursor, noMore]);

  return (
    <>
      {game && (
        <Layout padding={2}>
          <GameCardCommon
            game={game}
            hideTags={!!tags?.length}
            showMoreDetails
            srcSetSizes={[
              {
                mediaCondition:
                  '(max-width: 540px) and (orientation: portrait)',
                size: '47vw',
              },
              { size: '24vw' },
            ]}
          />
        </Layout>
      )}
      {tags && (
        <Layout padding={{ bottom: 1, x: 2 }}>
          <ActiveTagFilter
            gameAlias={gameAlias}
            page={TagPage.Stream}
            query={query}
          />
        </Layout>
      )}
      {streams.length === 0 ? (
        <NoChannelsLive
          game={game?.displayName ?? gameAlias}
          hasTags={!!tags?.length}
        />
      ) : (
        <ChannelList
          hideTags={!!tags?.length}
          loadMore={loadMore}
          streams={streams}
        />
      )}
    </>
  );
};

TopChannelsForGameBase.displayName = 'TopChannelsForGameBase';

export const TopChannelsForGame = createRefetchContainer(
  TopChannelsForGameBase,
  {
    query: graphql`
      fragment TopChannelsForGame_query on Query {
        ...ActiveTagFilter_query @include(if: $hasTagId)
        game(name: $gameAlias) {
          name
          displayName
          ...GameCardCommon_game
          streams(first: $channelsCount, after: $cursor, tags: $tags) {
            edges {
              cursor
              node {
                id
                ...DirectoryStreamCard_stream
              }
            }
          }
        }
      }
    `,
  },
  graphql`
    query TopChannelsForGame_RefetchQuery(
      $gameAlias: String!
      $channelsCount: Int!
      $cursor: Cursor
      $hasTagId: Boolean!
      $tagId: ID!
      $tags: [String!]
    ) {
      ...TopChannelsForGame_query
    }
  `,
);
