import type { FC } from 'react';
import { useCallback } from 'react';
import type { RelayRefetchProp } from 'react-relay/legacy';
import { createRefetchContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { useRefetchList } from 'tachyon-relay';
import { Color, CoreText, FontSize, Layout } from 'twitch-core-ui';
import { CHANNELS_PAGE_SIZE } from '../../../../config';
import { ActiveTagFilter } from '../../../common';
import { ChannelList, NoChannelsLive } from '../../../discovery';
import type { TopChannels_query } from './__generated__/TopChannels_query.graphql';

export type TopChannelsProps = {
  query: TopChannels_query;
  relay: RelayRefetchProp;
  tags: readonly string[] | null | undefined;
};

export const TopChannelsBase: FC<TopChannelsProps> = ({
  query,
  relay,
  tags,
}) => {
  const { formatMessage } = useIntl();
  const {
    endCursor,
    noMore,
    nodes: streams,
  } = useRefetchList(query.streams?.edges);

  const loadMore = useCallback(() => {
    if (noMore) {
      return;
    }

    relay.refetch((fragmentVariables) => ({
      ...fragmentVariables,
      channelsCount: CHANNELS_PAGE_SIZE * 3,
      cursor: endCursor,
    }));
  }, [relay, endCursor, noMore]);

  if (streams.length === 0) {
    return <NoChannelsLive />;
  }

  return (
    <>
      <Layout padding={{ top: 2, x: 2 }}>
        <CoreText
          bold
          breakpointMedium={{ fontSize: FontSize.Size1 }}
          color={Color.Base}
          fontSize={FontSize.Size2}
        >
          {formatMessage('Live Channels', 'TopChannelsBase')}
        </CoreText>
        {tags && (
          <Layout padding={{ y: 1 }}>
            <ActiveTagFilter page={TagPage.Browse} query={query} />
          </Layout>
        )}
      </Layout>
      <ChannelList
        hideTags={!!tags?.length}
        loadMore={loadMore}
        streams={streams}
      />
    </>
  );
};

TopChannelsBase.displayName = 'TopChannelsBase';

export const TopChannels = createRefetchContainer(
  TopChannelsBase,
  {
    query: graphql`
      fragment TopChannels_query on Query {
        ...ActiveTagFilter_query @include(if: $hasTagId)
        streams(first: $channelsCount, after: $cursor, tags: $tags) {
          edges {
            cursor
            node {
              id
              ...DirectoryStreamCard_stream
            }
          }
        }
      }
    `,
  },
  graphql`
    query TopChannels_RefetchQuery(
      $channelsCount: Int!
      $cursor: Cursor
      $hasTagId: Boolean!
      $tagId: ID!
      $tags: [String!]
    ) {
      ...TopChannels_query
    }
  `,
);
