import type { FC } from 'react';
import { useCallback } from 'react';
import type { RelayRefetchProp } from 'react-relay/legacy';
import { createRefetchContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { TagPage } from 'tachyon-discovery';
import { ExtendInteractionMedium } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { useRefetchList } from 'tachyon-relay';
import { Color, CoreText, FontSize, Layout } from 'twitch-core-ui';
import { GAMES_PAGE_SIZE } from '../../../../config';
import { ActiveTagFilter, InfiniteList } from '../../../common';
import { DirectoryGameCard } from '../../../discovery';
import type { GameList_query } from './__generated__/GameList_query.graphql';

export const ScGameList = styled(InfiniteList)`
  display: flex;
  flex-flow: wrap row;
  padding: 0.5%;
  width: 100%;

  @media (max-width: 540px) and (orientation: portrait) {
    padding: 1%;
  }
`;

export type GameListProps = {
  query: GameList_query;
  relay: RelayRefetchProp;
  tags: readonly string[] | null | undefined;
};

export const GameListBase: FC<GameListProps> = ({ query, relay, tags }) => {
  const { formatMessage } = useIntl();
  const {
    endCursor,
    noMore,
    nodes: games,
  } = useRefetchList(query.games?.edges);

  const loadMore = useCallback(() => {
    if (noMore) {
      return;
    }

    relay.refetch((fragmentVariables) => ({
      ...fragmentVariables,
      cursor: endCursor,
      gamesCount: GAMES_PAGE_SIZE * 3,
    }));
  }, [relay, endCursor, noMore]);

  const itemRenderer = useCallback(
    (idx: number): JSX.Element => (
      <DirectoryGameCard
        game={games[idx]}
        hideTags={!!tags?.length}
        key={games[idx].id}
        position={idx}
      />
    ),
    [games, tags],
  );

  return (
    <ExtendInteractionMedium value="game_list">
      <Layout padding={{ top: 2, x: 2 }}>
        <CoreText
          bold
          breakpointMedium={{ fontSize: FontSize.Size1 }}
          color={Color.Base}
          fontSize={FontSize.Size2}
        >
          {formatMessage('Browse', 'GameListBase')}
        </CoreText>
        {tags && (
          <Layout padding={{ y: 1 }}>
            <ActiveTagFilter page={TagPage.Browse} query={query} />
          </Layout>
        )}
      </Layout>
      <ScGameList
        itemRenderer={itemRenderer}
        length={games.length}
        loadMore={loadMore}
        pageSize={GAMES_PAGE_SIZE}
      />
    </ExtendInteractionMedium>
  );
};
GameListBase.displayName = 'GameListBase';

export const GameList = createRefetchContainer(
  GameListBase,
  {
    query: graphql`
      fragment GameList_query on Query {
        ...ActiveTagFilter_query @include(if: $hasTagId)
        games(first: $gamesCount, after: $cursor, tags: $tags) {
          edges {
            cursor
            node {
              id
              ...DirectoryGameCard_game
            }
          }
        }
      }
    `,
  },
  graphql`
    query GameList_RefetchQuery(
      $gamesCount: Int!
      $cursor: Cursor
      $hasTagId: Boolean!
      $tagId: ID!
      $tags: [String!]
    ) {
      ...GameList_query
    }
  `,
);
