import { Component } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { ItemPage } from 'tachyon-discovery';
import { horizontalOverflowIntoSafeArea } from 'tachyon-more-ui';
import { Layout } from 'twitch-core-ui';
import { ShelfGameCard } from '../ShelfGameCard';
import { ShelfStreamCard } from '../ShelfStreamCard';
import { ShelfTitle } from '../ShelfTitle';
import { filterShelfContent } from '../utils';
import type { Shelf_shelf } from './__generated__/Shelf_shelf.graphql';

export type { Shelf_shelf } from './__generated__/Shelf_shelf.graphql';

// horizontalOverflowIntoSafeArea({ buffer: '1rem' }): 1rem comes from the
// parent Layout having 1rem of padding
const ScShelfContainer = styled.div`
  display: flex;
  ${horizontalOverflowIntoSafeArea({ buffer: '1rem' })}
  padding-bottom: 2rem;
`;

type ShelfQueryProps = {
  rowPosition: number;
  shelf: Shelf_shelf;
};

type ShelfContainerState = {
  /** The set of content edges that are valid and will be rendered */
  filteredEdges: NonNullable<Shelf_shelf['content']['edges']>;
};

export type ShelfProps = ShelfQueryProps;

export class ShelfBase extends Component<ShelfProps> {
  public static displayName = 'Shelf';

  public state: ShelfContainerState = {
    filteredEdges: [],
  };

  public static getDerivedStateFromProps(
    nextProps: ShelfProps,
  ): Partial<ShelfContainerState> | null {
    if (nextProps.shelf.content.edges) {
      return {
        filteredEdges: filterShelfContent(nextProps.shelf),
      };
    }
    return null;
  }

  public render(): JSX.Element | null {
    if (this.state.filteredEdges.length === 0) {
      return null;
    }

    return (
      <Layout margin={{ y: 1 }}>
        <Layout margin={{ bottom: 0.5 }}>
          <ShelfTitle title={this.props.shelf.title} />
        </Layout>
        <ScShelfContainer>{this.renderShelfItems()}</ScShelfContainer>
      </Layout>
    );
  }

  private renderShelfItems(): (JSX.Element | null)[] | null {
    return this.state.filteredEdges.map((edge, index) => {
      if (!edge.node) {
        return null;
      }

      let card: JSX.Element;
      if (edge.node.__typename === 'Stream') {
        card = (
          <ShelfStreamCard
            shelfTracking={this.getShelfTracking(index, edge.trackingID)}
            stream={edge.node}
          />
        );
      } else if (edge.node.__typename === 'Game') {
        card = (
          <ShelfGameCard
            game={edge.node}
            shelfTracking={this.getShelfTracking(index, edge.trackingID)}
          />
        );
      } else {
        return null;
      }

      return (
        <Layout
          flexShrink={0}
          key={index}
          padding={{
            /* right-pad even the last card to ensure ending boundary */
            right: 1,
          }}
        >
          {card}
        </Layout>
      );
    });
  }

  private getShelfTracking(index: number, trackingID: string) {
    return {
      itemPage: ItemPage.Discover,
      itemPosition: index,
      itemTrackingID: trackingID,
      reasonTarget: this.props.shelf.trackingInfo.reasonTarget,
      reasonTargetType: this.props.shelf.trackingInfo.reasonTargetType,
      reasonType: this.props.shelf.trackingInfo.reasonType,
      rowName: this.props.shelf.trackingInfo.rowName,
      rowPosition: this.props.rowPosition,
      section: ItemPage.Discover,
    };
  }
}

export const Shelf = createFragmentContainer(ShelfBase, {
  shelf: graphql`
    fragment Shelf_shelf on Shelf {
      __typename
      id
      title {
        ...ShelfTitle_title
      }
      content {
        __typename
        edges {
          __typename
          trackingID
          node {
            __typename
            ... on Game {
              ...ShelfGameCard_game
            }
            ... on Stream {
              broadcaster {
                login
              }
              ...ShelfStreamCard_stream
            }
          }
        }
      }
      trackingInfo {
        __typename
        rowName
        reasonType
        reasonTarget
        reasonTargetType
      }
    }
  `,
});
