import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { mockStreamCardQuery } from '../../../../discovery/test-mocks';
import { ShelfGameCard } from '../ShelfGameCard';
import { ShelfStreamCard } from '../ShelfStreamCard';
import type { Shelf_shelf } from './__generated__/Shelf_shelf.graphql';
import { ShelfBase } from '.';

describe(ShelfBase, () => {
  function mockShelfContainerQuery(options?: {
    edges: Shelf_shelf['content']['edges'];
  }): Shelf_shelf {
    return {
      ' $refType': 'Shelf_shelf',
      __typename: 'Shelf',
      content: {
        __typename: 'Content',
        edges: options?.edges ?? null,
      },
      id: '192355F010B8FC891A76089685D1A49F5E93D34D',
      title: {
        ' $fragmentRefs': { ShelfTitle_title: true },
      },
      trackingInfo: {
        __typename: 'TrackingInfo',
        reasonTarget: null,
        reasonTargetType: null,
        reasonType: 'popularity',
        rowName: 'TopLiveChannels',
      },
    };
  }

  const setup = createShallowWrapperFactory(ShelfBase, () => ({
    rowPosition: 0,
    shelf: mockShelfContainerQuery(),
  }));

  it('renders nothing if there are no shelves', () => {
    const { wrapper } = setup();
    expect(wrapper.isEmptyRender()).toBeTruthy();
  });

  it('renders StreamCard if edge type is "Stream"', () => {
    const { wrapper } = setup({
      shelf: mockShelfContainerQuery({
        edges: [
          {
            __typename: 'Node',
            node: {
              ' $fragmentRefs': { ShelfStreamCard_stream: true },
              __typename: 'Stream',
              broadcaster: {
                login: 'ishbu',
              },
            },
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd5193275f6',
          },
        ],
      }),
    });
    expect(wrapper.find(ShelfStreamCard)).toExist();
    expect(wrapper.find(ShelfGameCard)).not.toExist();
  });

  it('renders GameCard if edge type is "Game"', () => {
    const { wrapper } = setup({
      shelf: mockShelfContainerQuery({
        edges: [
          {
            __typename: 'Node',
            node: {
              ' $fragmentRefs': { ShelfGameCard_game: true },
              __typename: 'Game',
            },
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd5193275f6',
          },
        ],
      }),
    });
    expect(wrapper.find(ShelfStreamCard)).not.toExist();
    expect(wrapper.find(ShelfGameCard)).toExist();
  });

  it('does not render Stream edges if broadcaster is null', () => {
    const { wrapper } = setup({
      shelf: mockShelfContainerQuery({
        edges: [
          {
            __typename: 'Node',
            node: {
              ' $fragmentRefs': { ShelfStreamCard_stream: true },
              __typename: 'Stream',
              broadcaster: null,
            },
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd5193275f6',
          },
        ],
      }),
    });
    expect(wrapper.find(ShelfStreamCard)).not.toExist();
  });

  it('does not render empty nodes', () => {
    const { wrapper } = setup({
      shelf: mockShelfContainerQuery({
        edges: [
          {
            __typename: 'Node',
            node: {
              ' $fragmentRefs': { ShelfGameCard_game: true },
              __typename: 'Game',
            },
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd5193275f6',
          },
          {
            __typename: 'Node',
            node: null,
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd575f6',
          },
          {
            __typename: 'Node',
            node: {
              ' $fragmentRefs': { ShelfGameCard_game: true },
              __typename: 'Game',
            },
            trackingID: '75141b7d-34cc-48ac-bd9d-efs3193275f6',
          },
        ],
      }),
    });
    expect(wrapper.find(ShelfGameCard)).toHaveLength(2);
  });

  it('does not render nodes of type "%other"', () => {
    const { wrapper } = setup({
      shelf: mockShelfContainerQuery({
        edges: [
          {
            __typename: 'Node',
            node: {
              __typename: '%other',
              ...mockStreamCardQuery(),
            },
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd5193275f6',
          },
        ],
      }),
    });
    expect(wrapper.find(ShelfGameCard)).not.toExist();
    expect(wrapper.find(ShelfStreamCard)).not.toExist();
  });

  it('renders nothing if there are no valid content edges', () => {
    const { wrapper } = setup({
      shelf: mockShelfContainerQuery({
        edges: [
          {
            __typename: 'Node',
            node: {
              ' $fragmentRefs': { ShelfStreamCard_stream: true },
              __typename: 'Stream',
              broadcaster: null,
            },
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd5193275f6',
          },
          {
            __typename: 'Node',
            node: null,
            trackingID: '09141b7d-34cc-48ac-bd9d-ebd575f6',
          },
        ],
      }),
    });
    expect(wrapper.isEmptyRender()).toBeTruthy();
  });
});
