import type { FC } from 'react';
import { useMemo } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import type { RecommendationTrackingParams } from 'tachyon-discovery';
import { ContentType, useDiscoveryTracking } from 'tachyon-discovery';
import { safeAreaWidthInVw } from 'tachyon-more-ui';
import { convertToUnsafeID } from 'tachyon-relay';
import type { ImpressionListenerProps } from 'tachyon-utils';
import {
  uniqueIDGenerator,
  useEffectOnce,
  withImpressionListener,
} from 'tachyon-utils';
import type { GameCardCommonProps } from '../../../../discovery';
import { GameCardCommon } from '../../../../discovery';
import type { ShelfGameCard_game } from './__generated__/ShelfGameCard_game.graphql';

export type { ShelfGameCard_game } from './__generated__/ShelfGameCard_game.graphql';

const SHELF_BOX_ART_SIZES = [
  {
    size: '27vw',
  },
  {
    mediaCondition: '(max-width: 540px) and (orientation: portrait)',
    size: '36vw',
  },
  {
    mediaCondition: '(min-width: 768px) and (orientation: landscape)',
    size: '20vw',
  },
];

const ScShelfGameCardWrapper = styled.div`
  ${safeAreaWidthInVw({ vw: 27 })}

  @media (max-width: 540px) and (orientation: portrait) {
    ${safeAreaWidthInVw({ vw: 36 })}
  }

  @media (min-width: 768px) and (orientation: landscape) {
    ${safeAreaWidthInVw({ vw: 20 })}
  }
`;

export interface ShelfGameCardProps
  extends Omit<GameCardCommonProps, 'game' | 'srcSetSizes'>,
    ImpressionListenerProps {
  game: ShelfGameCard_game;
  shelfTracking: Omit<
    RecommendationTrackingParams,
    'contentType' | 'itemID' | 'modelTrackingID'
  >;
}

export const ShelfGameCardBase: FC<ShelfGameCardProps> = (props) => {
  const { onRecommendationItemClick, onRecommendationItemDisplay } =
    useDiscoveryTracking();
  const { game, impressionListener, shelfTracking } = props;

  const trackingParams = useMemo<RecommendationTrackingParams>(() => {
    const itemID = convertToUnsafeID(game.id);
    // TODO: should this actually be broadcaster.id?
    const itemTrackingID =
      shelfTracking.itemTrackingID ?? uniqueIDGenerator(16);
    const modelTrackingID = shelfTracking.itemTrackingID;

    return {
      ...shelfTracking,
      contentType: ContentType.Game,
      itemID,
      itemTrackingID,
      modelTrackingID,
    };
  }, [shelfTracking, game.id]);

  useEffectOnce(() => {
    impressionListener.registerImpressionCallback(() => {
      onRecommendationItemDisplay(trackingParams);
    });
  });

  return (
    <ScShelfGameCardWrapper>
      <GameCardCommon
        {...props}
        onClick={() => {
          onRecommendationItemClick(trackingParams);
        }}
        srcSetSizes={SHELF_BOX_ART_SIZES}
      />
    </ScShelfGameCardWrapper>
  );
};

ShelfGameCardBase.displayName = 'ShelfGameCardBase';

export const ShelfGameCard = createFragmentContainer(
  withImpressionListener(ShelfGameCardBase),
  {
    game: graphql`
      fragment ShelfGameCard_game on Game {
        id
        name
        ...GameCardCommon_game
      }
    `,
  },
);
