import type { FC } from 'react';
import { useMemo } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import type { RecommendationTrackingParams } from 'tachyon-discovery';
import { ContentType, useDiscoveryTracking } from 'tachyon-discovery';
import { safeAreaWidthInVw } from 'tachyon-more-ui';
import { convertToUnsafeID } from 'tachyon-relay';
import type { ImpressionListenerProps } from 'tachyon-utils';
import {
  uniqueIDGenerator,
  useEffectOnce,
  withImpressionListener,
} from 'tachyon-utils';
import type { StreamCardCommonPublicProps } from '../../../../discovery';
import { StreamCardCommon } from '../../../../discovery';
import { isStreamRenderable } from '../utils';
import type { ShelfStreamCard_stream } from './__generated__/ShelfStreamCard_stream.graphql';

export type { ShelfStreamCard_stream } from './__generated__/ShelfStreamCard_stream.graphql';

const SHELF_THUMBNAIL_SIZES = [
  {
    size: '39.5vw',
  },
  {
    mediaCondition: '(max-width: 540px) and (orientation: portrait)',
    size: '65vw',
  },
  {
    mediaCondition: '(min-width: 768px) and (orientation: landscape)',
    size: '26vw',
  },
];

const ScShelfStreamCardWrapper = styled.div`
  ${safeAreaWidthInVw({ vw: 39.5 })}

  @media (max-width: 540px) and (orientation: portrait) {
    ${safeAreaWidthInVw({ vw: 65 })}
  }

  @media (min-width: 768px) and (orientation: landscape) {
    ${safeAreaWidthInVw({ vw: 26 })}
  }
`;

export interface ShelfStreamCardProps
  extends Omit<StreamCardCommonPublicProps, 'srcSetSizes'>,
    ImpressionListenerProps {
  shelfTracking: Omit<
    RecommendationTrackingParams,
    'contentType' | 'itemID' | 'modelTrackingID'
  >;
  stream: ShelfStreamCard_stream;
}

export const ShelfStreamCardBase: FC<ShelfStreamCardProps> = (props) => {
  const { onRecommendationItemClick, onRecommendationItemDisplay } =
    useDiscoveryTracking();
  const { impressionListener, shelfTracking, stream } = props;

  const trackingParams = useMemo<RecommendationTrackingParams>(() => {
    const itemID = convertToUnsafeID(stream.id);
    const itemTrackingID =
      shelfTracking.itemTrackingID ?? uniqueIDGenerator(16);
    const modelTrackingID = shelfTracking.itemTrackingID;

    return {
      ...shelfTracking,
      contentType: ContentType.LiveStream,
      itemID,
      itemTrackingID,
      modelTrackingID,
    };
  }, [shelfTracking, stream.id]);

  useEffectOnce(() => {
    impressionListener.registerImpressionCallback(() => {
      onRecommendationItemDisplay(trackingParams);
    });
  });

  if (!isStreamRenderable(stream)) {
    return null;
  }

  return (
    <ScShelfStreamCardWrapper>
      <StreamCardCommon
        {...props}
        onClick={() => {
          onRecommendationItemClick(trackingParams);
        }}
        srcSetSizes={SHELF_THUMBNAIL_SIZES}
      />
    </ScShelfStreamCardWrapper>
  );
};

ShelfStreamCardBase.displayName = 'ShelfStreamCardBase';

export const ShelfStreamCard = createFragmentContainer(
  withImpressionListener(ShelfStreamCardBase),
  {
    stream: graphql`
      fragment ShelfStreamCard_stream on Stream {
        id
        ...StreamCardCommon_stream
        broadcaster {
          login
        }
      }
    `,
  },
);
