import { Component } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import {
  Color,
  CoreLink,
  CoreText,
  FontSize,
  TextTransform,
  TextType,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import type { ShelfTitle_title } from './__generated__/ShelfTitle_title.graphql';

interface ShelfTitleQueryProps {
  title: ShelfTitle_title;
}

export interface ShelfTitleProps extends ShelfTitleQueryProps {}

type TitleToken = ShelfTitle_title['localizedTitleTokens'][0];

export class ShelfTitleBase extends Component<ShelfTitleProps> {
  public static displayName = 'ShelfTitle';

  public render(): JSX.Element {
    const { title } = this.props;

    const parsedTokens = title.localizedTitleTokens.map((token) =>
      this.parseTitleToken(token),
    );
    const hasInvalidTokens = parsedTokens.some((token) => !token);

    if (hasInvalidTokens || parsedTokens.length === 0) {
      return (
        <CoreText
          bold
          breakpointMedium={{ fontSize: FontSize.Size3 }}
          color={Color.Alt2}
          fontSize={FontSize.Size6}
          key={title.key}
          transform={TextTransform.Uppercase}
          type={TextType.Span}
        >
          {title.fallbackLocalizedTitle}
        </CoreText>
      );
    }

    return <>{parsedTokens}</>;
  }

  private parseTitleToken(token: TitleToken): JSX.Element | null {
    const node = token?.node;

    if (!node) {
      return null;
    }

    if (node.__typename === 'Game') {
      return (
        <CoreLink
          key={node.displayName}
          linkTo="/deferToRenderLink"
          onClick={this.trackTitleClick}
          renderLink={renderTachyonLink({
            route: RouteName.GameDirectory,
            routeParams: { gameAlias: node.displayName },
          })}
        >
          {this.renderTitle(node.displayName)}
        </CoreLink>
      );
    } else if (node.__typename === 'TextToken') {
      if (node.location === 'CATEGORIES_DIRECTORY') {
        return (
          <CoreLink
            key={node.text}
            linkTo="/deferToRenderLink"
            onClick={this.trackTitleClick}
            renderLink={renderTachyonLink({
              route: RouteName.GamesDirectory,
            })}
          >
            {this.renderTitle(node.text)}
          </CoreLink>
        );
      } else if (node.location === 'STREAMS_DIRECTORY') {
        return (
          <CoreLink
            key={node.text}
            linkTo="/deferToRenderLink"
            onClick={this.trackTitleClick}
            renderLink={renderTachyonLink({
              route: RouteName.GameDirectoryAll,
            })}
          >
            {this.renderTitle(node.text)}
          </CoreLink>
        );
      }
      return (
        <CoreText
          bold
          breakpointMedium={{ fontSize: FontSize.Size3 }}
          color={Color.Alt2}
          fontSize={FontSize.Size6}
          key={node.text}
          transform={TextTransform.Uppercase}
          type={TextType.Span}
        >
          {node.text}
        </CoreText>
      );
    }

    return null;
  }

  private renderTitle = (text: string): JSX.Element => (
    <CoreText
      bold
      breakpointMedium={{ fontSize: FontSize.Size3 }}
      fontSize={FontSize.Size6}
      transform={TextTransform.Uppercase}
      type={TextType.Span}
    >
      {text}
    </CoreText>
  );

  private trackTitleClick = (): void => {
    // TODO: is this needed / tracked?
    // track clicks here
  };
}

export const ShelfTitle = createFragmentContainer(ShelfTitleBase, {
  title: graphql`
    fragment ShelfTitle_title on ShelfTitle {
      key
      fallbackLocalizedTitle
      localizedTitleTokens {
        node {
          ... on Game {
            __typename
            id
            name
            displayName
          }
          ... on TextToken {
            __typename
            text
            hasEmphasis
            location
          }
        }
      }
    }
  `,
});
