import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { CoreLink, CoreText } from 'twitch-core-ui';
import type { ShelfTitle_title } from './__generated__/ShelfTitle_title.graphql';
import { ShelfTitleBase } from '.';

describe(ShelfTitleBase, () => {
  function mockShelfTitleQuery(options?: { tokens: any }): ShelfTitle_title {
    return {
      ' $refType': 'ShelfTitle_title',
      fallbackLocalizedTitle: 'Popular categories',
      key: 'TopGames',
      localizedTitleTokens: options?.tokens || [],
    };
  }

  const setup = createShallowWrapperFactory(ShelfTitleBase, () => ({
    title: mockShelfTitleQuery(),
  }));

  it('renders fallback title if there are no localized tokens', () => {
    const { props, wrapper } = setup();
    expect(wrapper.find(CoreText).props().children).toBe(
      props.title.fallbackLocalizedTitle,
    );
  });

  it('renders fallback title if any localized tokens are invalid', () => {
    const { props, wrapper } = setup({
      title: mockShelfTitleQuery({
        tokens: [
          {
            node: null,
          },
        ],
      }),
    });
    expect(wrapper.find(CoreText).props().children).toBe(
      props.title.fallbackLocalizedTitle,
    );
  });

  it('renders link to game directory if token is of type "Game"', () => {
    const { props, wrapper } = setup({
      title: mockShelfTitleQuery({
        tokens: [
          {
            node: {
              __typename: 'Game',
              displayName: 'League of Legends',
              id: '21779',
              name: 'League of Legends',
            },
          },
        ],
      }),
    });

    expect(wrapper.find(CoreText).props().children).toBe(
      (props.title.localizedTitleTokens[0]!.node as any).displayName,
    );
  });

  it('renders link to categories directory if location is of type "CATEGORIES_DIRECTORY"', () => {
    const { props, wrapper } = setup({
      title: mockShelfTitleQuery({
        tokens: [
          {
            node: {
              __typename: 'TextToken',
              hasEmphasis: true,
              location: 'CATEGORIES_DIRECTORY',
              text: 'categories',
            },
          },
        ],
      }),
    });

    expect(wrapper.find(CoreText).props().children).toBe(
      (props.title.localizedTitleTokens[0]!.node! as any).text,
    );
  });

  it('renders link to streams directory if location is of type "STREAMS_DIRECTORY"', () => {
    const { props, wrapper } = setup({
      title: mockShelfTitleQuery({
        tokens: [
          {
            node: {
              __typename: 'TextToken',
              hasEmphasis: true,
              location: 'STREAMS_DIRECTORY',
              text: 'channels',
            },
          },
        ],
      }),
    });

    expect(wrapper.find(CoreText).props().children).toBe(
      (props.title.localizedTitleTokens[0]!.node! as any).text,
    );
  });

  it('renders non-link text correctly', () => {
    const { props, wrapper } = setup({
      title: mockShelfTitleQuery({
        tokens: [
          {
            node: {
              __typename: 'TextToken',
              hasEmphasis: false,
              location: 'NO_LINK',
              text: 'Popular',
            },
          },
          {
            node: {
              __typename: 'TextToken',
              hasEmphasis: false,
              location: 'NO_LINK',
              text: 'live',
            },
          },
          {
            node: {
              __typename: 'TextToken',
              hasEmphasis: false,
              location: 'NO_LINK',
              text: 'channels',
            },
          },
        ],
      }),
    });
    const coreTextElem = wrapper.find(CoreText);

    expect(wrapper.find(CoreLink)).not.toExist();
    expect(coreTextElem.at(0).props().children).toBe(
      (props.title.localizedTitleTokens[0]!.node! as any).text,
    );
    expect(coreTextElem.at(1).props().children).toBe(
      (props.title.localizedTitleTokens[1]!.node! as any).text,
    );
    expect(coreTextElem.at(2).props().children).toBe(
      (props.title.localizedTitleTokens[2]!.node! as any).text,
    );
  });
});
