import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  Button,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  SVG,
  SVGAsset,
  SVGType,
  TextType,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { Shelf } from './Shelf';
import type { RecommendationShelves_query } from './__generated__/RecommendationShelves_query.graphql';

export type RecommendationShelvesProps = {
  query: RecommendationShelves_query;
};

export const RecommendationShelvesBase: FC<RecommendationShelvesProps> = ({
  query,
}) => {
  const { formatMessage } = useIntl();
  const shelves = query.shelves?.edges;

  if (!shelves || shelves.length === 0) {
    return (
      <Layout
        alignItems={AlignItems.Center}
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        fullHeight
        fullWidth
        justifyContent={JustifyContent.Center}
        padding={{ bottom: 5 }}
      >
        <Layout padding={{ bottom: 1 }}>
          <SVG
            asset={SVGAsset.DeadGlitch}
            height={100}
            type={SVGType.Alt2}
            width={100}
          />
        </Layout>
        <CoreText fontSize={FontSize.Size4}>
          {formatMessage('Something went wrong...', 'RecommendationShelves')}
        </CoreText>
        <CoreText color={Color.Alt2} fontSize={FontSize.Size5}>
          {formatMessage(
            "We couldn't find your recommendations",
            'RecommendationShelves',
          )}
        </CoreText>
        <Layout padding={{ top: 1 }}>
          <Button
            linkTo="/deferToRenderLink"
            renderLink={renderTachyonLink({
              route: RouteName.GamesDirectory,
            })}
          >
            {formatMessage('Browse games', 'RecommendationShelves')}
          </Button>
        </Layout>
      </Layout>
    );
  }

  return (
    <Layout padding={{ top: 2, x: 1 }}>
      <CoreText
        bold
        breakpointMedium={{ fontSize: FontSize.Size1 }}
        color={Color.Base}
        fontSize={FontSize.Size2}
        type={TextType.H1}
      >
        {formatMessage('Welcome to Twitch!', 'RecommendationShelves')}
      </CoreText>
      {shelves.map((shelf, index) => (
        <Shelf key={index} rowPosition={index} shelf={shelf.node} />
      ))}
    </Layout>
  );
};

RecommendationShelvesBase.displayName = 'RecommendationShelves';

export const RecommendationShelves = createFragmentContainer(
  RecommendationShelvesBase,
  {
    query: graphql`
      fragment RecommendationShelves_query on Query {
        shelves(
          requestID: $requestID
          platform: $platform
          first: $limit
          itemsPerRow: $itemsPerRow
        ) {
          edges {
            node {
              ...Shelf_shelf
            }
          }
        }
      }
    `,
  },
);
