import { useState } from 'react';
import { graphql, useMutation } from 'react-relay/hooks';
import { useCurrentUser } from 'tachyon-auth';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import type {
  ReportContentMetadata,
  ReportContentType,
} from 'tachyon-report-wizard';
import { ReportWizardRoot } from 'tachyon-report-wizard';
import { flattenHeaderOrParam, isBrowser } from 'tachyon-utils';
import { Display, JustifyContent, Layout } from 'twitch-core-ui';
import { Page } from '../../layouts';
import type { PageWithQueryProps, TomorrowPage } from '../types';
import { ReportWizardError } from './ReportWizardError';
import { WizardComponent } from './WizardComponent';
import type {
  ReportWizardPage_QueryResponse,
  ReportWizardPage_QueryVariables,
} from './__generated__/ReportWizardPage_Query.graphql';
import type { ReportWizard_Block_User_Mutation } from './__generated__/ReportWizard_Block_User_Mutation.graphql';
import type {
  ReportContentErrorCode,
  ReportWizard_Create_Report_Mutation,
} from './__generated__/ReportWizard_Create_Report_Mutation.graphql';
import type { ReportWizard_Unblock_User_Mutation } from './__generated__/ReportWizard_Unblock_User_Mutation.graphql';
import {
  CHANNEL_ID_KEY,
  COLLECTION_ID_KEY,
  CONTENT_ID_KEY,
  CONTENT_TYPE_KEY,
  IS_EMBED_KEY,
  LIVE_UP_MESSAGE_KEY,
  MEDIA_TIMESTAMP_KEY,
  ReportSubmitStatus,
  SQUAD_ID_KEY,
  VOD_ID_KEY,
  reportWizardGetInitialProps,
} from './utils';

export interface ReportContext {
  contentID?: string;
  contentMetadata?: ReportContentMetadata;
  contentType?: ReportContentType;
}

type ReportWizardPageInitialProps = {
  queryVariables: ReportWizardPage_QueryVariables;
};

type ReportWizardPageProps = PageWithQueryProps<
  ReportWizardPage_QueryResponse,
  ReportWizardPageInitialProps
>;

export const ReportWizardPage: TomorrowPage<
  ReportWizardPageInitialProps,
  ReportWizardPageProps,
  ReportWizardPage_QueryVariables
> = ({ currentUser, loading, queryVariables, reportWizard, targetUser }) => {
  const { currentQuery } = useRouterUtils();
  const { loggedIn } = useCurrentUser();

  // Handle Blocking and Unblocking TargetUser
  const targetUserAlreadyBlocked =
    !!currentUser &&
    !!currentUser.blockedUsers.some((u) => u?.id === targetUser?.id);
  const [targetUserBlocked, setTargetUserBlocked] = useState<boolean>(
    targetUserAlreadyBlocked,
  );

  const [commitBlockUser] =
    useMutation<ReportWizard_Block_User_Mutation>(graphql`
      mutation ReportWizard_Block_User_Mutation($input: BlockUserInput!) {
        blockUser(input: $input) {
          targetUser {
            id
          }
        }
      }
    `);
  const [commitUnblockUser] =
    useMutation<ReportWizard_Unblock_User_Mutation>(graphql`
      mutation ReportWizard_Unblock_User_Mutation($input: UnblockUserInput!) {
        unblockUser(input: $input) {
          targetUser {
            id
          }
        }
      }
    `);

  const block = () => {
    if (targetUser?.id && currentUser) {
      commitBlockUser({
        onCompleted(_, err) {
          if (!err) {
            setTargetUserBlocked(true);
          }
        },
        variables: {
          input: { targetUserID: targetUser.id },
        },
      });
    }
  };
  const unblock = () => {
    if (targetUser?.id && currentUser) {
      commitUnblockUser({
        onCompleted(_, err) {
          if (!err) {
            setTargetUserBlocked(false);
          }
        },
        variables: {
          input: { targetUserID: targetUser.id },
        },
      });
    }
  };
  const onToggleBlock = targetUserBlocked ? unblock : block;

  // Keep track of report creation status
  const [reportSubmitStatus, setReportSubmitStatus] =
    useState<ReportSubmitStatus>(ReportSubmitStatus.Unsubmitted);
  const [reportCreationErrorCode, setReportCreationErrorCode] =
    useState<ReportContentErrorCode>();
  const [commitCreateReport, reportBeingCreated] =
    useMutation<ReportWizard_Create_Report_Mutation>(graphql`
      mutation ReportWizard_Create_Report_Mutation(
        $input: ReportUserContentInput!
      ) {
        reportUserContent(input: $input) {
          error {
            code
          }
          contentID
        }
      }
    `);

  // isEmbed represents whether this component is being used in a native app webview
  const isEmbed = flattenHeaderOrParam(currentQuery[IS_EMBED_KEY]) === 'true';

  // We will we re-direct unauthenticated users to Twilight because TMW does not have login implemented yet
  if (isBrowser() && !loggedIn) {
    // TODO: Change this to router.replace({ route: RouteName.Login }) once TMW gets login
    window.location.replace(
      `https://www.twitch.com/${queryVariables.login}/report`,
    );
    return null;
  }

  // Handle error in fetching report wizard data, currentUser or targetUser
  if (!targetUser?.id || !currentUser?.id || !reportWizard) {
    return (
      <Page hideTopNav={isEmbed} loading={loading}>
        <ReportWizardError />
      </Page>
    );
  }

  // validate input from query params
  const reportContext: ReportContext = {
    contentID: flattenHeaderOrParam(currentQuery[CONTENT_ID_KEY]),
    contentMetadata: {
      channelID: flattenHeaderOrParam(currentQuery[CHANNEL_ID_KEY]),
      collectionID: flattenHeaderOrParam(currentQuery[COLLECTION_ID_KEY]),
      liveUpMessage: flattenHeaderOrParam(currentQuery[LIVE_UP_MESSAGE_KEY]),
      mediaTimestamp: flattenHeaderOrParam(currentQuery[MEDIA_TIMESTAMP_KEY]),
      squadID: flattenHeaderOrParam(currentQuery[SQUAD_ID_KEY]),
      vodID: flattenHeaderOrParam(currentQuery[VOD_ID_KEY]),
    },
    contentType: flattenHeaderOrParam(
      currentQuery[CONTENT_TYPE_KEY],
    ) as ReportContentType,
  };

  return (
    <Page hideTopNav={isEmbed} loading={loading}>
      <Layout display={Display.Flex} justifyContent={JustifyContent.Center}>
        <Layout margin={{ bottom: 1 }} padding={2}>
          <ReportWizardRoot
            content={reportContext.contentType}
            contentID={reportContext.contentID}
            contentMetadata={reportContext.contentMetadata}
            data={reportWizard}
            fromUserID={currentUser.id}
            hideBlockPrompt={targetUserAlreadyBlocked}
            onCloseWizard={() => {
              if (isEmbed) {
                // for native apps, we'll want to close the webview here
                // TODO: Test this in native app
                window.close();
                // TODO: Figure out if we can/want to pass any data back to native appsStashed changes
              }
            }}
            onEvent={() => {
              // TODO: Implement analytics
              // spadeClient.call({
              //   ...event,
              //   reportSessionID.current,
              // });
            }}
            onSubmit={(formValues) => {
              commitCreateReport({
                onCompleted(resp, err) {
                  if (err) {
                    setReportCreationErrorCode(
                      resp.reportUserContent?.error?.code,
                    );
                  } else {
                    setReportSubmitStatus(ReportSubmitStatus.Success);
                    setReportCreationErrorCode(undefined);
                  }
                },
                onError(_) {
                  setReportSubmitStatus(ReportSubmitStatus.MutationError);
                },
                variables: {
                  input: {
                    ...formValues,
                    // we grab the unique reportSessionID generated for the query for tracking report wizard sessions
                    sessionID: queryVariables.reportSessionID,
                  },
                },
              });
            }}
            onToggleBlock={onToggleBlock}
            reportSubmissionError={reportCreationErrorCode}
            reportSubmissionLoading={reportBeingCreated}
            reportSubmissionSuccess={
              reportSubmitStatus === ReportSubmitStatus.Success
            }
            targetUserBlocked={targetUserBlocked}
            targetUserID={targetUser.id}
            targetUserUsername={targetUser.login}
          >
            <WizardComponent />
          </ReportWizardRoot>
        </Layout>
      </Layout>
    </Page>
  );
};

ReportWizardPage.displayName = 'ReportWizardPage';
ReportWizardPage.pageviewTracking = defaultPageviewTracking;
ReportWizardPage.requiresJsForInteractivity = true;
ReportWizardPage.handlesLoading = true;
ReportWizardPage.getInitialProps = reportWizardGetInitialProps;

ReportWizardPage.query = graphql`
  query ReportWizardPage_Query(
    $login: String!
    $url: String!
    $content: [ReportContentType!]
    $reportSessionID: ID!
  ) {
    targetUser: user(login: $login) {
      id
      login
      displayName
    }
    currentUser {
      id
      blockedUsers {
        id
      }
      email
      login
    }
    reportWizard(content: $content, reportSessionID: $reportSessionID) {
      reasons {
        id
        countryCode
        disclosureText
        toSAndCountryReasons {
          id
          isApplicableToCountryRegulations
          text
          description
          detailedReasons {
            id
            title
            deadEndType
          }
        }
      }
      reportableContent {
        id
        type
        applicableReasons {
          id
          visibility
          reportReason {
            # do not bloat the query response size by adding fields here
            # we only grab id here to match reasons in applicableReasons & toSAndCountryReasons
            id
          }
        }
        deadEndType
      }
    }
    ...PageHead_query
  }
`;
