import { internet } from 'faker';
import { act } from 'react-dom/test-utils';
import {
  DescriptionForm,
  Reason,
  ReportWizardRoot,
} from 'tachyon-report-wizard';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { Button } from 'twitch-core-ui';
import { mockRouter } from '../../../routing/test-mocks';
import { ReportWizardError } from './ReportWizardError';
import { WizardComponent } from './WizardComponent';
import { mockReportWizardQuery } from './test-mocks';
import { ReportWizardPage } from '.';

// mock routing
jest.mock('../../../routing', () => ({
  ...jest.requireActual('../../../routing'),
  useTachyonRouter: () => mockRouter,
}));
const mockCommit = jest.fn();
// mock create report mutation
jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useMutation: jest.fn(() => [mockCommit, false]),
}));
// mock current user hook
let mockUseCurrentUser = { loggedIn: true };
jest.mock('tachyon-auth', () => ({
  ...jest.requireActual('tachyon-auth'),
  useCurrentUser: () => mockUseCurrentUser,
}));

describe(ReportWizardPage, () => {
  const setup = createMountWrapperFactory(ReportWizardPage, () => {
    return {
      ...mockReportWizardQuery,
      loading: false,
      login: 'login',
      queryVariables: {
        content: undefined,
        login: 'login',
        reportSessionID: 'some-unique-uuid',
        url: internet.url(),
      },
    };
  });

  it('loads report wizard root component as expected', () => {
    mockUseCurrentUser = { loggedIn: true };
    const { wrapper } = setup();
    expect(wrapper.find(ReportWizardError)).not.toExist();
    expect(wrapper.find(ReportWizardRoot)).toExist();
  });

  it('loads report wizard error when query is missing targetUser in response', () => {
    mockUseCurrentUser = { loggedIn: true };
    const { wrapper } = setup({ targetUser: undefined });
    expect(wrapper.find(ReportWizardError)).toExist();
    expect(wrapper.find(ReportWizardRoot)).not.toExist();
  });

  it('loads report wizard error when query is missing reportWizard data in response', () => {
    mockUseCurrentUser = { loggedIn: true };
    const { wrapper } = setup({ reportWizard: null });
    expect(wrapper.find(ReportWizardError)).toExist();
    expect(wrapper.find(ReportWizardRoot)).not.toExist();
  });

  it('redirects user to Twilight if user is not authenticated on Tomorrow', () => {
    // mock window.location.replace
    const windowLocationReplaceMock = jest.fn();
    Object.defineProperty(window, 'location', {
      value: { replace: windowLocationReplaceMock },
      writable: true,
    });

    mockUseCurrentUser = { loggedIn: false };
    setup({ reportWizard: null });
    expect(windowLocationReplaceMock).toHaveBeenCalled();
    expect(windowLocationReplaceMock).toHaveBeenCalledWith(
      'https://www.twitch.com/login/report',
    );
  });

  it('calls submit mutation when the submit report button is clicked', () => {
    mockUseCurrentUser = { loggedIn: true };
    mockCommit.mockImplementationOnce(({ onCompleted }) => {
      onCompleted();
    });

    const { wrapper } = setup();
    expect(wrapper.find(WizardComponent)).toExist();

    // select report reason
    expect(wrapper.find(Reason)).toExist();
    act(() => {
      wrapper.find(Button).first().simulate('click');
    });
    wrapper.update();

    // fill out description
    act(() => {
      wrapper.find(DescriptionForm).props().onChange('a valid description');
    });

    // click submit button
    act(() => {
      wrapper.find(Button).last().simulate('click');
    });

    // expect mutation to have been invoked
    expect(mockCommit).toHaveBeenCalledTimes(1);
  });

  it('expect onCompleted and onError to to be called after submit mutation fires which updates report wizard root', () => {
    mockUseCurrentUser = { loggedIn: true };
    mockCommit.mockImplementationOnce(({ onCompleted, onError }) => {
      onCompleted(
        { reportUserContent: { error: { code: 'INTERNAL_ERROR' } } },
        true,
      );
      onError();
    });

    const { wrapper } = setup();
    expect(wrapper.find(WizardComponent)).toExist();

    // select report reason
    expect(wrapper.find(Reason)).toExist();
    act(() => {
      wrapper.find(Button).first().simulate('click');
    });
    wrapper.update();

    // fill out description
    act(() => {
      wrapper.find(DescriptionForm).props().onChange('a valid description');
    });

    // click submit button
    act(() => {
      wrapper.find(Button).last().simulate('click');
    });
    wrapper.update();

    // expect mutation to have been invoked
    expect(mockCommit).toHaveBeenCalledTimes(1);
    expect(wrapper.find(ReportWizardRoot).props().reportSubmissionLoading).toBe(
      false,
    );
    expect(wrapper.find(ReportWizardRoot).props().reportSubmissionError).toBe(
      'INTERNAL_ERROR',
    );
    expect(wrapper.find(ReportWizardRoot).props().reportSubmissionSuccess).toBe(
      false,
    );
  });

  it('expect mutation success to update report wizard root accordingly', () => {
    mockUseCurrentUser = { loggedIn: true };
    mockCommit.mockImplementationOnce(({ onCompleted }) => {
      onCompleted({ reportUserContent: { id: '123' } }, false);
    });

    const { wrapper } = setup();
    expect(wrapper.find(WizardComponent)).toExist();

    // select report reason
    expect(wrapper.find(Reason)).toExist();
    act(() => {
      wrapper.find(Button).first().simulate('click');
    });
    wrapper.update();

    // fill out description
    act(() => {
      wrapper.find(DescriptionForm).props().onChange('a valid description');
    });

    // click submit button
    act(() => {
      wrapper.find(Button).last().simulate('click');
    });
    wrapper.update();

    // expect mutation to have been invoked
    expect(mockCommit).toHaveBeenCalledTimes(1);
    expect(wrapper.find(ReportWizardRoot).props().reportSubmissionLoading).toBe(
      false,
    );
    expect(
      wrapper.find(ReportWizardRoot).props().reportSubmissionError,
    ).toBeUndefined();
    expect(wrapper.find(ReportWizardRoot).props().reportSubmissionSuccess).toBe(
      true,
    );
  });
});
