import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { SearchSuggestionType } from 'tachyon-discovery';
import {
  StreamType,
  StreamTypeIndicator,
  getBoxArtSrcAndSrcSet,
} from 'tachyon-more-ui';
import type { ImpressionListenerProps } from 'tachyon-utils';
import { useEffectOnce, withImpressionListener } from 'tachyon-utils';
import {
  AlignItems,
  Aspect,
  AspectRatio,
  Avatar,
  BorderRadius,
  CoreImage,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  FontSize,
  FontWeight,
  JustifyContent,
  Layout,
  SVG,
  SVGAsset,
  SVGType,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import { getSearchLink } from '../../utils';
import type { SearchItem_suggestion } from './__generated__/SearchItem_suggestion.graphql';

export interface SearchItemProps extends ImpressionListenerProps {
  onClick: (
    suggestionType: SearchSuggestionType,
    clickedItemID: string,
  ) => void;
  onDisplay: (suggestionType: SearchSuggestionType) => void;
  suggestion: SearchItem_suggestion | null;
  text: string;
}

export const ScSearchItem = styled(Layout)`
  height: 5rem;
`;

export const ScSearchItemIcon = styled(Layout)`
  width: 4rem;
`;

export const SearchItemBase: FC<SearchItemProps> = ({
  impressionListener,
  onClick,
  onDisplay,
  suggestion,
  text,
}) => {
  let icon: JSX.Element | null = null;
  let verified: JSX.Element | null = null;
  let liveIndicator: JSX.Element | null = null;
  let fontWeight: FontWeight = FontWeight.Regular;
  let suggestionType: SearchSuggestionType;
  let contentID = '';

  let link = getSearchLink(text).renderLink;

  switch (suggestion?.__typename) {
    case 'SearchSuggestionCategory': {
      suggestionType = SearchSuggestionType.Category;
      contentID = suggestion.id;
      fontWeight = FontWeight.SemiBold;
      if (suggestion.game) {
        link = renderTachyonLink({
          route: RouteName.GameDirectory,
          routeParams: { gameAlias: suggestion.game.name },
        });
      }
      icon = (
        <Aspect ratio={AspectRatio.BoxArt}>
          <CoreImage
            alt={text}
            {...getBoxArtSrcAndSrcSet(suggestion.boxArtURL)}
          />
        </Aspect>
      );
      break;
    }
    case 'SearchSuggestionChannel': {
      suggestionType = SearchSuggestionType.Live;
      contentID = suggestion.id;
      fontWeight = FontWeight.SemiBold;
      link = renderTachyonLink({
        route: RouteName.Channel,
        routeParams: { login: suggestion.login },
      });
      icon = (
        <Avatar
          alt={text}
          borderRadius={BorderRadius.Rounded}
          size={40}
          src={suggestion.profileImageURL}
          userLogin={suggestion.login}
        />
      );
      if (suggestion.isVerified) {
        verified = (
          <SVG
            asset={SVGAsset.Verified}
            height={18}
            type={SVGType.Brand}
            width={18}
          />
        );
      }

      // channels are guarenteed to be live in suggestions
      liveIndicator = (
        <StreamTypeIndicator
          isOverlay={false}
          isPulsing={false}
          streamType={StreamType.Live}
        />
      );
      break;
    }
    default:
      suggestionType = SearchSuggestionType.Text;
      icon = <SVG asset={SVGAsset.NavSearch} />;
  }

  useEffectOnce(() => {
    impressionListener.registerImpressionCallback(() => {
      onDisplay(suggestionType);
    });
  });

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      onClick={() => {
        onClick(suggestionType, contentID);
      }}
      renderLink={link}
      variant={CoreLinkType.Inherit}
    >
      <ScSearchItem alignItems={AlignItems.Center} display={Display.Flex}>
        <ScSearchItemIcon
          display={Display.Flex}
          flexShrink={0}
          justifyContent={JustifyContent.Center}
          margin={{ right: 1 }}
        >
          {icon}
        </ScSearchItemIcon>
        <CoreText ellipsis fontSize={FontSize.Size4} fontWeight={fontWeight}>
          {text}
        </CoreText>
        <Layout display={Display.Flex} flexGrow={1} margin={{ x: 0.5 }}>
          {verified}
        </Layout>
        {liveIndicator}
      </ScSearchItem>
    </CoreLink>
  );
};

SearchItemBase.displayName = 'SearchItemBase';

export const SearchItem = createFragmentContainer(
  withImpressionListener(SearchItemBase),
  {
    suggestion: graphql`
      fragment SearchItem_suggestion on SearchSuggestionContent {
        __typename
        ... on SearchSuggestionChannel {
          id
          isVerified
          login
          profileImageURL(width: 50)
          user {
            id
            stream {
              id
              game {
                id
              }
            }
          }
        }
        ... on SearchSuggestionCategory {
          id
          boxArtURL
          game {
            name
          }
        }
      }
    `,
  },
);
