import type { FC } from 'react';
import { useEffect, useRef } from 'react';
import type { Disposable } from 'react-relay/hooks';
import { graphql } from 'react-relay/hooks';
import type { SearchSuggestionType } from 'tachyon-discovery';
import { SearchQueryType, useDiscoveryTracking } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { useRefetchableFragmentWithoutSuspense } from 'tachyon-relay';
import {
  reduceToNonNullNodes,
  uniqueIDGenerator,
  useDebounce,
} from 'tachyon-utils';
import {
  AlignItems,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  FontSize,
  Layout,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { ScSearchItem, SearchItem } from './SearchItem';
import type { SearchAutocomplete_suggestions$key } from './__generated__/SearchAutocomplete_suggestions.graphql';

const REFETCH_DEBOUNCE_MS = 150;
const ID_LENGTH = 16;

// argumentDefinitions are necessary here to allow Relay to synthesize the
// correct refetch based on our custom variable for the skip directive
const searchAutocompleteFragment = graphql`
  fragment SearchAutocomplete_suggestions on Query
  @argumentDefinitions(
    queryFragment: { type: "String!" }
    requestID: { type: "ID" }
    noQueryFragment: { type: "Boolean!" }
  )
  @refetchable(queryName: "SearchAutocomplete_RefetchQuery") {
    searchSuggestions(queryFragment: $queryFragment, requestID: $requestID)
      @skip(if: $noQueryFragment) {
      edges {
        node {
          id
          text
          content {
            ...SearchItem_suggestion
          }
        }
      }
      tracking {
        modelTrackingID
        responseID
      }
    }
  }
`;

// remove once concurrent mode is ready
const searchAutocompleteQuery = graphql`
  query SearchAutocomplete_Query(
    $queryFragment: String!
    $noQueryFragment: Boolean!
    $requestID: ID
  ) {
    ...SearchAutocomplete_suggestions
      @arguments(
        queryFragment: $queryFragment
        requestID: $requestID
        noQueryFragment: $noQueryFragment
      )
  }
`;

export type SearchAutocompleteProps = {
  hide: () => void;
  searchTerm: string;
  suggestions: SearchAutocomplete_suggestions$key;
};

export const SearchAutocomplete: FC<SearchAutocompleteProps> = ({
  hide,
  searchTerm,
  suggestions: suggestionsRef,
}) => {
  const [suggestions, refetch] = useRefetchableFragmentWithoutSuspense(
    searchAutocompleteFragment,
    suggestionsRef,
    searchAutocompleteQuery,
  );

  const {
    onSearchQuerySubmit,
    onSearchSuggestionClick,
    onSearchSuggestionDisplay,
    onSearchSuggestionRequest,
  } = useDiscoveryTracking();
  const requestID = useRef(uniqueIDGenerator(ID_LENGTH));
  const refetchDisposable = useRef<Disposable | null>(null);
  const { formatMessage } = useIntl();

  function updateSearchSuggestions(queryFragment: string) {
    if (refetchDisposable.current) {
      refetchDisposable.current.dispose();
    }

    requestID.current = uniqueIDGenerator(ID_LENGTH);
    refetchDisposable.current = refetch(
      {
        noQueryFragment: queryFragment === '',
        queryFragment,
        requestID: requestID.current,
      },
      {
        onComplete: () => {
          refetchDisposable.current = null;
        },
      },
    );

    if (queryFragment !== '') {
      onSearchSuggestionRequest({
        query: queryFragment,
        requestID: requestID.current,
      });
    }
  }

  const updateSearchSuggestionsDebounced = useDebounce(
    updateSearchSuggestions,
    REFETCH_DEBOUNCE_MS,
    { leading: true, trailing: true },
  );

  useEffect(() => {
    // we fire a suggestions request immediately even though the UI isn't on
    // screen to provide instant suggestions should the user be dissatisfied
    // with any pre-rendered full results
    updateSearchSuggestionsDebounced(searchTerm);
  }, [searchTerm, updateSearchSuggestionsDebounced]);

  const searchItems = reduceToNonNullNodes(
    suggestions.searchSuggestions?.edges,
  );
  // add a search suggestion for the search term. this will just be a
  // clickable suggestion for the search query that the user is typing
  searchItems.push({
    content: null,
    id: uniqueIDGenerator(),
    text: searchTerm,
  });

  if (!searchTerm) {
    return null;
  }

  return (
    <ul>
      {searchItems.map((node, idx) => {
        const trackingParams = {
          itemPosition: idx,
          modelTrackingID:
            suggestions.searchSuggestions?.tracking?.modelTrackingID,
          requestID: requestID.current,
          responseID: suggestions.searchSuggestions?.tracking?.responseID,
          suggestionText: node.text,
          suggestionTrackingID: node.id,
        };
        return (
          <Layout as="li" key={node.id} padding={{ x: 1, y: 0.5 }}>
            <SearchItem
              onClick={(
                suggestionType: SearchSuggestionType,
                clickedItemID: string,
              ) => {
                onSearchSuggestionClick({
                  ...trackingParams,
                  clickedItemID,
                  suggestionType,
                });
                onSearchQuerySubmit({
                  query: searchTerm,
                  suggestionPosition: idx,
                  suggestionTrackingID: trackingParams.suggestionTrackingID,
                  type: SearchQueryType.Suggestion,
                });
                hide();
              }}
              onDisplay={(suggestionType: SearchSuggestionType) => {
                onSearchSuggestionDisplay({
                  ...trackingParams,
                  suggestionType,
                });
              }}
              suggestion={node.content ?? null}
              text={node.text}
            />
          </Layout>
        );
      })}
      <CoreLink
        linkTo="/deferToRenderLink"
        renderLink={renderTachyonLink({
          route: RouteName.Channel,
          routeParams: { login: searchTerm },
        })}
        variant={CoreLinkType.Inherit}
      >
        <ScSearchItem
          alignItems={AlignItems.Center}
          display={Display.Flex}
          padding={{ x: 1, y: 0.5 }}
        >
          <CoreText ellipsis fontSize={FontSize.Size4}>
            {formatMessage(
              'Go to {name}',
              { name: searchTerm },
              'SearchOverlay',
            )}
          </CoreText>
        </ScSearchItem>
      </CoreLink>
    </ul>
  );
};

SearchAutocomplete.displayName = 'SearchAutocomplete';
