import { datatype, lorem } from 'faker';
import { useRefetchableFragmentWithoutSuspense } from 'tachyon-relay';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { ScSearchItem, SearchItem } from './SearchItem';
import { SearchAutocomplete } from '.';

jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  // eliminate timing component from testing
  useDebounce: jest.fn((fn) => fn),
}));

const mockRefetch = jest.fn();
jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  useRefetchableFragmentWithoutSuspense: jest.fn(() => [{}, mockRefetch]),
}));
const mockUseRefetchableFragmentWithoutSuspense =
  useRefetchableFragmentWithoutSuspense as jest.Mock;

const mockOnSearchInputFocus = jest.fn();
const mockOnSearchQuerySubmit = jest.fn();
const mockOnSearchSuggestionClick = jest.fn();
const mockOnSearchSuggestionDisplay = jest.fn();
const mockOnSearchSuggestionRequest = jest.fn();

// SearchItem is connected to relay so it otherwise prevents mount testing this component
jest.mock('./SearchItem', () => ({
  ...jest.requireActual('./SearchItem'),
  SearchItem: ({ onClick }: { onClick: any }) => <div onClick={onClick} />,
}));

jest.mock('tachyon-discovery', () => ({
  ...jest.requireActual('tachyon-discovery'),
  useDiscoveryTracking: () => ({
    onSearchInputFocus: mockOnSearchInputFocus,
    onSearchQuerySubmit: mockOnSearchQuerySubmit,
    onSearchSuggestionClick: mockOnSearchSuggestionClick,
    onSearchSuggestionDisplay: mockOnSearchSuggestionDisplay,
    onSearchSuggestionRequest: mockOnSearchSuggestionRequest,
  }),
}));

describe(SearchAutocomplete, () => {
  const suggestions = {
    ' $refType': 'SearchAutocomplete_suggestions',
    searchSuggestions: {
      edges: [
        {
          node: {
            content: {
              ' $fragmentRefs': { SearchItem_suggestion: true },
            },
            id: datatype.uuid(),
            text: lorem.word(),
          },
        },
      ],
      tracking: {
        modelTrackingID: datatype.uuid(),
        responseID: datatype.uuid(),
      },
    },
  };

  const setup = createMountWrapperFactory(SearchAutocomplete, () => ({
    hide: jest.fn(),
    searchTerm: '',
    suggestions: {} as any,
  }));

  it('empty renders when searchTerm is blank', () => {
    const { wrapper } = setup({ searchTerm: '' });
    expect(wrapper).toBeEmptyRender();
  });

  it('shows the go to item', () => {
    const { wrapper } = setup({ searchTerm: 'foo' });
    expect(wrapper.find(ScSearchItem)).toExist();
  });

  it('renders a suggestion for the search query', () => {
    mockUseRefetchableFragmentWithoutSuspense.mockImplementationOnce(() => [
      suggestions,
      mockRefetch,
    ]);

    const { props, wrapper } = setup({ searchTerm: 'test' });
    expect(wrapper.find(SearchItem)).toHaveLength(2);
    expect(wrapper.find(SearchItem).at(1).prop('text')).toEqual(
      props.searchTerm,
    );
    expect(mockRefetch).toHaveBeenCalledTimes(1);
  });

  it('calls onSearchSuggestionClick on item click', () => {
    const mockHide = jest.fn();
    const { wrapper } = setup({ hide: mockHide, searchTerm: 'test' });
    wrapper.find(SearchItem).at(0).simulate('click');

    expect(mockOnSearchSuggestionClick).toHaveBeenCalled();
    expect(mockOnSearchQuerySubmit).toHaveBeenCalled();
    expect(mockHide).toHaveBeenCalled();
  });

  it('refetches when the searchTerm changes', () => {
    const { wrapper } = setup({ searchTerm: 'test' });
    expect(mockRefetch).toHaveBeenCalledTimes(1);

    wrapper.setProps({ searchTerm: 'test1' });

    expect(mockRefetch).toHaveBeenCalledTimes(2);
  });
});
