import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { formatLiveViewersCount, getBoxArtSrcAndSrcSet } from 'tachyon-more-ui';
import type { ImpressionListenerProps } from 'tachyon-utils';
import { useEffectOnce, withImpressionListener } from 'tachyon-utils';
import {
  AlignItems,
  Color,
  CoreImage,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  InjectLayout,
  Layout,
  Overflow,
  TextType,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import { GameTagList } from '../../../../common';
import { ScImageContainer } from '../../utils';
import type { SearchCategoryCard_game } from './__generated__/SearchCategoryCard_game.graphql';

export interface SearchCategoryCardBaseProps extends ImpressionListenerProps {
  game: SearchCategoryCard_game;
  onClick: () => void;
  onDisplay: () => void;
}

export const SearchCategoryCardBase: FC<SearchCategoryCardBaseProps> = ({
  game,
  impressionListener,
  onClick,
  onDisplay,
}) => {
  const { boxArtURL, displayName, viewersCount } = game;
  const { formatMessage } = useIntl();
  const link = renderTachyonLink({
    route: RouteName.GameDirectory,
    routeParams: { gameAlias: game.name },
  });

  useEffectOnce(() => {
    impressionListener.registerImpressionCallback(onDisplay);
  });

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      onClick={onClick}
      renderLink={link}
      variant={CoreLinkType.Inherit}
    >
      <Layout alignItems={AlignItems.Center} display={Display.Flex}>
        <ScImageContainer display={Display.Flex} flexShrink={0}>
          <InjectLayout fullHeight>
            <CoreImage
              alt={displayName}
              {...getBoxArtSrcAndSrcSet(boxArtURL)}
            />
          </InjectLayout>
        </ScImageContainer>
        <Layout overflow={Overflow.Hidden} padding={{ left: 1 }}>
          <CoreText ellipsis type={TextType.H4}>
            {displayName}
          </CoreText>
          <CoreText color={Color.Alt2}>
            {viewersCount !== null &&
              formatLiveViewersCount(formatMessage, viewersCount)}
          </CoreText>
          <GameTagList game={game} page={TagPage.Browse} />
        </Layout>
      </Layout>
    </CoreLink>
  );
};

SearchCategoryCardBase.displayName = 'SearchCategoryCardBase';

export const SearchCategoryCard = createFragmentContainer(
  withImpressionListener(SearchCategoryCardBase),
  {
    game: graphql`
      fragment SearchCategoryCard_game on Game {
        name
        displayName
        boxArtURL
        viewersCount
        ...TagList_game
      }
    `,
  },
);
