import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { getPlayingChannel } from 'tachyon-page-utils';
import type { ImpressionListenerProps } from 'tachyon-utils';
import { useEffectOnce, withImpressionListener } from 'tachyon-utils';
import { CoreLink, CoreLinkType } from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import { SearchOfflineChannel } from '../../SearchOfflineChannel';
import { SearchStreamCard } from '../../SearchStreamCard';
import type { SearchChannelCard_channel } from './__generated__/SearchChannelCard_channel.graphql';

export interface SearchChannelCardBaseProps extends ImpressionListenerProps {
  channel: SearchChannelCard_channel;
  onClick: (isLive: boolean) => void;
  onDisplay: (isLive: boolean) => void;
}

export const SearchChannelCardBase: FC<SearchChannelCardBaseProps> = ({
  channel,
  impressionListener,
  onClick,
  onDisplay,
}) => {
  const playingChannel = getPlayingChannel(channel);
  const isLive = !!playingChannel;

  useEffectOnce(() => {
    impressionListener.registerImpressionCallback(() => {
      onDisplay(isLive);
    });
  });

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      onClick={() => {
        onClick(isLive);
      }}
      renderLink={renderTachyonLink({
        route: RouteName.Channel,
        routeParams: { login: channel.login },
      })}
      variant={CoreLinkType.Inherit}
    >
      {playingChannel ? (
        <SearchStreamCard
          isPartner={channel.roles?.isPartner}
          stream={playingChannel.stream}
        />
      ) : (
        <SearchOfflineChannel
          channel={channel}
          isPartner={channel.roles?.isPartner}
        />
      )}
    </CoreLink>
  );
};

SearchChannelCardBase.displayName = 'SearchChannelCardBase';

export const SearchChannelCard = createFragmentContainer(
  withImpressionListener(SearchChannelCardBase),
  {
    channel: graphql`
      fragment SearchChannelCard_channel on User {
        id
        login
        ...SearchOfflineChannel_channel
        stream {
          id
          ...SearchStreamCard_stream
        }
        roles {
          isPartner
        }
      }
    `,
  },
);
