import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import {
  StreamType,
  StreamTypeIndicator,
  formatLiveViewersCount,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  CardImage,
  CardImageSize,
  Color,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  FontSize,
  FontWeight,
  Layout,
  Overflow,
  Position,
  TextTransform,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import { SearchContentChannelName } from '../../SearchContentChannelName';
import type { SearchRelatedLiveChannels_channels } from './__generated__/SearchRelatedLiveChannels_channels.graphql';

export interface SearchRelatedLiveChannelsProps {
  channels: SearchRelatedLiveChannels_channels;
  searchTerm: string;
}

const ScRelatedChannelContainer = styled(Layout)`
  width: 16rem;
`;

export const SearchRelatedLiveChannelsBase: FC<SearchRelatedLiveChannelsProps> =
  ({ channels, searchTerm }) => {
    const { formatMessage } = useIntl();

    if (!channels.items?.length) {
      return null;
    }

    return (
      <>
        <Layout margin={{ bottom: 1 }}>
          <CoreText
            color={Color.Alt2}
            fontSize={FontSize.Size5}
            fontWeight={FontWeight.SemiBold}
            transform={TextTransform.Uppercase}
          >
            {formatMessage(
              'People searching for "{name}" also watch:',
              { name: searchTerm },
              'SearchResults',
            )}
          </CoreText>
        </Layout>
        <Layout display={Display.Flex} overflow={Overflow.Scroll}>
          {channels.items.map((channel) => {
            const broadcaster = channel.stream?.broadcaster;

            if (!broadcaster?.login) {
              return null;
            }

            return (
              <ScRelatedChannelContainer
                key={channel.id}
                margin={{ right: 0.5 }}
                position={Position.Relative}
              >
                <CoreLink
                  linkTo="/deferToRenderLink"
                  renderLink={renderTachyonLink({
                    route: RouteName.Channel,
                    routeParams: { login: broadcaster.login },
                  })}
                  variant={CoreLinkType.Inherit}
                >
                  <CardImage
                    size={CardImageSize.Size16}
                    {...getChannelThumbnailSrcAndSrcSet(
                      channel.stream?.previewImageURL,
                    )}
                    alt=""
                  />
                  <Layout
                    attachLeft
                    attachTop
                    margin={0.5}
                    position={Position.Absolute}
                  >
                    <StreamTypeIndicator
                      isOverlay
                      isPulsing={false}
                      streamType={StreamType.Live}
                    />
                  </Layout>
                  <SearchContentChannelName
                    displayName={getFormattedUserDisplayName(broadcaster)}
                    isPartner={broadcaster.roles?.isPartner}
                  />
                  {channel.stream?.game && (
                    <CoreText color={Color.Alt2} ellipsis>
                      {channel.stream.game.displayName}
                    </CoreText>
                  )}
                  {channel.stream?.viewersCount && (
                    <CoreText color={Color.Alt2}>
                      {formatLiveViewersCount(
                        formatMessage,
                        channel.stream.viewersCount,
                      )}
                    </CoreText>
                  )}
                </CoreLink>
              </ScRelatedChannelContainer>
            );
          })}
        </Layout>
      </>
    );
  };

SearchRelatedLiveChannelsBase.displayName = 'SearchRelatedChannelsBase';

export const SearchRelatedLiveChannels = createFragmentContainer(
  SearchRelatedLiveChannelsBase,
  {
    channels: graphql`
      fragment SearchRelatedLiveChannels_channels on SearchForResultRelatedLiveChannels {
        items {
          id
          stream {
            id
            viewersCount
            previewImageURL
            game {
              id
              name
              displayName
            }
            broadcaster {
              id
              login
              displayName
              roles {
                isPartner
              }
            }
          }
        }
      }
    `,
  },
);

SearchRelatedLiveChannels.displayName = 'SearchRelatedLiveChannels';
