import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import {
  formatLiveViewersCount,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import type { ImpressionListenerProps } from 'tachyon-utils';
import {
  formatVideoLength,
  useEffectOnce,
  withImpressionListener,
} from 'tachyon-utils';
import {
  AlignItems,
  CardImage,
  CardImageSize,
  Color,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  Layout,
  MediaCardStat,
  Overflow,
  Position,
  SVG,
  SVGAsset,
  SVGType,
  TextTransform,
  TextType,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import { ScImageContainer } from '../../utils';
import type { SearchVideoCard_video } from './__generated__/SearchVideoCard_video.graphql';

export interface SearchVideoCardBaseProps extends ImpressionListenerProps {
  onClick: () => void;
  onDisplay: () => void;
  video: SearchVideoCard_video;
}

export const SearchVideoCardBase: FC<SearchVideoCardBaseProps> = ({
  impressionListener,
  onClick,
  onDisplay,
  video,
}) => {
  const { formatMessage, formatNumberShort, formatPastRelativeDate } =
    useIntl();
  const {
    game,
    lengthSeconds,
    owner,
    previewThumbnailURL,
    publishedAt,
    title,
    viewCount,
  } = video;

  useEffectOnce(() => {
    impressionListener.registerImpressionCallback(onDisplay);
  });

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      onClick={onClick}
      renderLink={renderTachyonLink({
        interactionContent: 'next-video-overlay-link',
        interactionTargetPath: `/videos/${video.id}`,
        route: RouteName.Vod,
        routeParams: { videoID: video.id },
      })}
      variant={CoreLinkType.Inherit}
    >
      <Layout alignItems={AlignItems.Center} display={Display.Flex}>
        <ScImageContainer position={Position.Relative}>
          <CardImage
            size={CardImageSize.Size16}
            {...getChannelThumbnailSrcAndSrcSet(previewThumbnailURL)}
            alt={title ?? ''}
          />
          {lengthSeconds !== null && (
            <Layout
              attachLeft
              attachTop
              margin={0.5}
              position={Position.Absolute}
            >
              <MediaCardStat>{formatVideoLength(lengthSeconds)}</MediaCardStat>
            </Layout>
          )}
        </ScImageContainer>
        <Layout overflow={Overflow.Hidden} padding={{ left: 1 }}>
          <CoreText ellipsis type={TextType.H4}>
            {title}
          </CoreText>
          <Layout display={Display.Flex}>
            <CoreText color={Color.Alt2}>{owner?.displayName}</CoreText>
            {owner?.roles?.isPartner && (
              <Layout padding={{ left: 0.5 }}>
                <SVG
                  asset={SVGAsset.Verified}
                  height={14}
                  type={SVGType.Brand}
                  width={14}
                />
              </Layout>
            )}
          </Layout>
          <CoreText color={Color.Alt2} transform={TextTransform.Capitalize}>
            {game?.displayName}
          </CoreText>
          <div>
            {viewCount !== null && (
              <CoreText color={Color.Alt2} ellipsis type={TextType.Span}>
                {formatLiveViewersCount(
                  formatMessage,
                  viewCount,
                  formatNumberShort(viewCount, {
                    maximumSignificantDigits: 3,
                  }),
                )}
              </CoreText>
            )}
            {viewCount !== null && publishedAt && <span> &#183; </span>}
            {publishedAt && (
              <CoreText color={Color.Alt2} ellipsis type={TextType.Span}>
                {formatPastRelativeDate(new Date(publishedAt))}
              </CoreText>
            )}
          </div>
        </Layout>
      </Layout>
    </CoreLink>
  );
};

SearchVideoCardBase.displayName = 'SearchVideoCardBase';

export const SearchVideoCard = createFragmentContainer(
  withImpressionListener(SearchVideoCardBase),
  {
    video: graphql`
      fragment SearchVideoCard_video on Video {
        publishedAt
        owner {
          id
          displayName
          login
          roles {
            isPartner
          }
        }
        id
        game {
          id
          name
          displayName
        }
        lengthSeconds
        previewThumbnailURL
        title
        viewCount
      }
    `,
  },
);
