import type { ComponentProps, FC } from 'react';
import { useCallback } from 'react';
import type { RelayRefetchProp } from 'react-relay/legacy';
import {
  createFragmentContainer,
  createRefetchContainer,
  graphql,
} from 'react-relay/legacy';
import type { FragmentRefs } from 'relay-runtime';
import { SearchCategory, useDiscoveryTracking } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import type { GraphQLNode } from 'tachyon-relay';
import { useRefetchList } from 'tachyon-relay';
import type { GQLEdges } from 'tachyon-utils';
import {
  exhaustedCase,
  reduceToNonNullNodes,
  uniqueIDGenerator,
} from 'tachyon-utils';
import { Layout } from 'twitch-core-ui';
import { InfiniteList } from '../../../common';
import { NoResultsFoundPage } from '../NoResultsFoundPage';
import { SearchContentLabel } from '../SearchContentLabel';
import type { Search_QueryResponse } from '../__generated__/Search_Query.graphql';
import { SearchType, getSearchLink } from '../utils';
import { SearchCategoryCard } from './SearchCategoryCard';
import { SearchChannelCard } from './SearchChannelCard';
import { SearchRelatedLiveChannels } from './SearchRelatedLiveChannels';
import { SearchVideoCard } from './SearchVideoCard';
import type { SearchContent_channels } from './__generated__/SearchContent_channels.graphql';
import type { SearchContent_games } from './__generated__/SearchContent_games.graphql';
import type { SearchContent_overview } from './__generated__/SearchContent_overview.graphql';
import type { SearchContent_videos } from './__generated__/SearchContent_videos.graphql';

const LOAD_MORE_LIMIT = 10;

type SearchForResult<Node> =
  | {
      cursor: string | null;
      items: ReadonlyArray<Node> | null;
    }
  | null
  | undefined;

/**
 * The SearchFor GraphQL response does not conform to the GraphQL Cursor Connections Specification. The 'edges' query lacks 'cursor', and the node type is a union of all possible types instead of the scoped query: eg when scoped under 'videos' or 'channels' nodes is typed as the full union set. The SearForResult response is processed to conform to the connections interface.
 */
function normalizeSearchFor<Node>(
  searchForResult: SearchForResult<Node>,
): GQLEdges<Node> {
  return searchForResult?.items?.map((item: Node) => ({
    cursor: searchForResult.cursor,
    node: item,
  }));
}

type CardProps<T> = T & {
  idx: number;
  searchTerm: string;
};

type ChannelCardProps = CardProps<{
  channel: ComponentProps<typeof SearchChannelCard>['channel'] & { id: string };
}>;

export const ChannelCard: FC<ChannelCardProps> = ({
  channel,
  idx,
  searchTerm,
}): JSX.Element => {
  const { onSearchResultClick, onSearchResultImpression } =
    useDiscoveryTracking();

  const trackingParams = {
    contentID: channel.id,
    itemIndex: idx,
    query: searchTerm,
    srpItemTrackingID: uniqueIDGenerator(),
  };

  return (
    <Layout padding={{ y: 0.5 }}>
      <SearchChannelCard
        channel={channel}
        onClick={(isLive) => {
          onSearchResultClick({
            ...trackingParams,
            contentType: isLive ? SearchCategory.Streams : SearchCategory.Users,
            isLive,
            itemRow: 0,
            subSection: isLive ? SearchCategory.Streams : SearchCategory.Users,
          });
        }}
        onDisplay={(isLive) => {
          onSearchResultImpression({
            ...trackingParams,
            isLive,
            itemRow: 0,
            subSection: isLive ? SearchCategory.Streams : SearchCategory.Users,
            subSectionPosition: 0,
          });
        }}
      />
    </Layout>
  );
};
ChannelCard.displayName = 'ChannelCard';

type CategoryCardProps = CardProps<{
  game: ComponentProps<typeof SearchCategoryCard>['game'] & { id: string };
}>;

export const CategoryCard: FC<CategoryCardProps> = ({
  game,
  idx,
  searchTerm,
}): JSX.Element => {
  const { onSearchResultClick, onSearchResultImpression } =
    useDiscoveryTracking();

  const trackingParams = {
    contentID: game.id,
    itemIndex: idx,
    query: searchTerm,
    srpItemTrackingID: uniqueIDGenerator(),
  };

  return (
    <Layout padding={{ y: 0.5 }}>
      <SearchCategoryCard
        game={game}
        onClick={() => {
          onSearchResultClick({
            ...trackingParams,
            contentType: SearchCategory.Games,
            itemRow: 1,
            subSection: SearchCategory.Games,
          });
        }}
        onDisplay={() => {
          onSearchResultImpression({
            ...trackingParams,
            itemRow: 1,
            subSection: SearchCategory.Games,
            subSectionPosition: 1,
          });
        }}
      />
    </Layout>
  );
};
CategoryCard.displayName = 'CategoryCard';

type VideoCardProps = CardProps<{
  video: ComponentProps<typeof SearchVideoCard>['video'] & { id: string };
}>;

export const VideoCard: FC<VideoCardProps> = ({
  idx,
  searchTerm,
  video,
}): JSX.Element => {
  const { onSearchResultClick, onSearchResultImpression } =
    useDiscoveryTracking();

  const trackingParams = {
    contentID: video.id,
    itemIndex: idx,
    query: searchTerm,
    srpItemTrackingID: uniqueIDGenerator(),
  };

  return (
    <Layout margin={{ y: 0.5 }}>
      <SearchVideoCard
        onClick={() => {
          onSearchResultClick({
            ...trackingParams,
            contentType: SearchCategory.Videos,
            itemRow: 2,
            subSection: SearchCategory.Videos,
          });
        }}
        onDisplay={() => {
          onSearchResultImpression({
            ...trackingParams,
            itemRow: 2,
            subSection: SearchCategory.Videos,
            subSectionPosition: 2,
          });
        }}
        video={video}
      />
    </Layout>
  );
};
VideoCard.displayName = 'VideoCard';

type InfiniteCardListBaseProps<Node extends GraphQLNode> = {
  content: SearchForResult<Node>;
  itemRenderer: (idx: number, node: Node) => JSX.Element;
  relay: RelayRefetchProp;
  searchType: 'CHANNEL' | 'GAME' | 'VOD';
};

export function InfiniteCardListBase<Node extends GraphQLNode>({
  content,
  itemRenderer,
  relay,
  searchType,
}: InfiniteCardListBaseProps<Node>): JSX.Element {
  const { endCursor, noMore, nodes } = useRefetchList(
    normalizeSearchFor(content),
  );

  const renderer = useCallback(
    (idx: number) => itemRenderer(idx, nodes[idx]),
    [nodes, itemRenderer],
  );

  const loadMore = useCallback(() => {
    if (noMore) {
      return;
    }

    relay.refetch((fragmentVariables) => ({
      ...fragmentVariables,
      target: {
        cursor: endCursor,
        index: searchType,
        limit: LOAD_MORE_LIMIT,
      },
    }));
  }, [relay, endCursor, noMore, searchType]);

  if (nodes.length === 0) {
    return <NoResultsFoundPage />;
  }

  return (
    <InfiniteList
      itemRenderer={renderer}
      length={nodes.length}
      loadMore={loadMore}
      pageSize={LOAD_MORE_LIMIT}
    />
  );
}
InfiniteCardListBase.displayName = 'InfiniteCardListBase';

type InfiniteChannelListBaseProps = {
  channels: SearchContent_channels;
  relay: RelayRefetchProp;
  searchTerm: string;
};
// istanbul ignore next: trivial
const InfiniteChannelListBase = ({
  channels,
  relay,
  searchTerm,
}: InfiniteChannelListBaseProps) => {
  const MemoizedChannelCard = useCallback(
    (
      idx: number,
      node: {
        readonly ' $fragmentRefs': FragmentRefs<'SearchChannelCard_channel'>;
        readonly id: string;
      },
    ) => (
      <ChannelCard
        channel={node}
        idx={idx}
        key={node.id}
        searchTerm={searchTerm}
      />
    ),
    [searchTerm],
  );
  return (
    <InfiniteCardListBase
      content={channels.searchFor?.channels}
      itemRenderer={MemoizedChannelCard}
      relay={relay}
      searchType="CHANNEL"
    />
  );
};
InfiniteChannelListBase.displayName = 'InfiniteChannelListBase';

// istanbul ignore next: trivial
const InfiniteChannelList = createRefetchContainer(
  InfiniteChannelListBase,
  {
    channels: graphql`
      fragment SearchContent_channels on Query {
        searchFor(userQuery: $userQuery, platform: $platform, target: $target)
          @skip(if: $noQuery) {
          channels {
            cursor
            items {
              id
              ...SearchChannelCard_channel
            }
          }
        }
      }
    `,
  },
  graphql`
    query SearchContent_ChannelRefetchQuery(
      $userQuery: String!
      $platform: String!
      $noQuery: Boolean!
      $target: SearchForTarget
    ) {
      ...SearchContent_channels @relay(mask: false)
    }
  `,
);
InfiniteChannelList.displayName = 'InfiniteChannelList';

type InfiniteCategoryListBaseProps = {
  games: SearchContent_games;
  relay: RelayRefetchProp;
  searchTerm: string;
};

// istanbul ignore next: trivial
const InfiniteCategoryListBase = ({
  games,
  relay,
  searchTerm,
}: InfiniteCategoryListBaseProps) => {
  const MemoizedCategoryCard = useCallback(
    (
      idx: number,
      node: {
        readonly ' $fragmentRefs': FragmentRefs<'SearchCategoryCard_game'>;
        readonly id: string;
      },
    ) => (
      <CategoryCard
        game={node}
        idx={idx}
        key={node.id}
        searchTerm={searchTerm}
      />
    ),
    [searchTerm],
  );
  return (
    <InfiniteCardListBase
      content={games.searchFor?.games}
      itemRenderer={MemoizedCategoryCard}
      relay={relay}
      searchType="GAME"
    />
  );
};
InfiniteCategoryListBase.displayName = 'InfiniteCategoryListBase';

// istanbul ignore next: trivial
const InfiniteCategoryList = createRefetchContainer(
  InfiniteCategoryListBase,
  {
    games: graphql`
      fragment SearchContent_games on Query {
        searchFor(userQuery: $userQuery, platform: $platform, target: $target)
          @skip(if: $noQuery) {
          games {
            cursor
            items {
              id
              ...SearchCategoryCard_game
            }
          }
        }
      }
    `,
  },
  graphql`
    query SearchContent_CategoryRefetchQuery(
      $userQuery: String!
      $platform: String!
      $noQuery: Boolean!
      $target: SearchForTarget
    ) {
      ...SearchContent_games @relay(mask: false)
    }
  `,
);
InfiniteCategoryList.displayName = 'InfiniteCategoryList';

type InfiniteVideoListBaseProps = {
  relay: RelayRefetchProp;
  searchTerm: string;
  videos: SearchContent_videos;
};

// istanbul ignore next: trivial
const InfiniteVideoListBase = ({
  relay,
  searchTerm,
  videos,
}: InfiniteVideoListBaseProps) => {
  const MemoizedVideoCard = useCallback(
    (
      idx: number,
      node: {
        readonly ' $fragmentRefs': FragmentRefs<'SearchVideoCard_video'>;
        readonly id: string;
      },
    ) => (
      <VideoCard idx={idx} key={node.id} searchTerm={searchTerm} video={node} />
    ),
    [searchTerm],
  );

  return (
    <InfiniteCardListBase
      content={videos.searchFor?.videos}
      itemRenderer={MemoizedVideoCard}
      relay={relay}
      searchType="VOD"
    />
  );
};
InfiniteVideoListBase.displayName = 'InfiniteVideoListBase';

// istanbul ignore next: trivial
const InfiniteVideoList = createRefetchContainer(
  InfiniteVideoListBase,
  {
    videos: graphql`
      fragment SearchContent_videos on Query {
        searchFor(userQuery: $userQuery, platform: $platform, target: $target)
          @skip(if: $noQuery) {
          videos {
            cursor
            items {
              id
              ...SearchVideoCard_video
            }
          }
        }
      }
    `,
  },
  graphql`
    query SearchContent_VideoRefetchQuery(
      $userQuery: String!
      $platform: String!
      $noQuery: Boolean!
      $target: SearchForTarget
    ) {
      ...SearchContent_videos @relay(mask: false)
    }
  `,
);
InfiniteVideoList.displayName = 'InfiniteVideoList';

type SearchOverviewBaseProps = {
  overview: SearchContent_overview;
  searchTerm: string;
};

export const SearchOverviewBase: FC<SearchOverviewBaseProps> = ({
  overview,
  searchTerm,
}) => {
  const { formatMessage } = useIntl();

  const channels = reduceToNonNullNodes(
    normalizeSearchFor(overview.searchFor?.channels),
  );
  const games = reduceToNonNullNodes(
    normalizeSearchFor(overview.searchFor?.games),
  );
  const videos = reduceToNonNullNodes(
    normalizeSearchFor(overview.searchFor?.videos),
  );
  const relatedLiveChannels = overview.searchFor?.relatedLiveChannels;

  if (
    !channels.length &&
    !games.length &&
    !videos.length &&
    !relatedLiveChannels
  ) {
    return <NoResultsFoundPage />;
  }

  return (
    <>
      {channels.length > 0 && (
        <Layout as="section" margin={{ bottom: 2 }}>
          <SearchContentLabel
            label={formatMessage('CHANNELS', 'SearchResults')}
            {...getSearchLink(searchTerm, SearchType.Channel)}
          />
          {channels.slice(0, 3).map((channel, idx) => (
            <ChannelCard
              channel={channel}
              idx={idx}
              key={channel.id}
              searchTerm={searchTerm}
            />
          ))}
        </Layout>
      )}
      {relatedLiveChannels && (
        <Layout as="section" margin={{ bottom: 2 }}>
          <SearchRelatedLiveChannels
            channels={relatedLiveChannels}
            searchTerm={searchTerm}
          />
        </Layout>
      )}
      {games.length > 0 && (
        <Layout as="section" margin={{ bottom: 2 }}>
          <SearchContentLabel
            label={formatMessage('CATEGORIES', 'SearchResults')}
            {...getSearchLink(searchTerm, SearchType.Category)}
          />
          {games.slice(0, 2).map((game, idx) => (
            <CategoryCard
              game={game}
              idx={idx}
              key={game.id}
              searchTerm={searchTerm}
            />
          ))}
        </Layout>
      )}
      {videos.length > 0 && (
        <Layout as="section" margin={{ bottom: 2 }}>
          <SearchContentLabel
            label={formatMessage('VIDEOS', 'SearchResults')}
            {...getSearchLink(searchTerm, SearchType.Video)}
          />
          {videos.slice(0, 2).map((video, idx) => (
            <VideoCard
              idx={idx}
              key={video.id}
              searchTerm={searchTerm}
              video={video}
            />
          ))}
        </Layout>
      )}
    </>
  );
};
SearchOverviewBase.displayName = 'SearchOverviewBase';

// istanbul ignore next: trivial
const SearchOverview = createFragmentContainer(SearchOverviewBase, {
  overview: graphql`
    fragment SearchContent_overview on Query {
      ...SearchContent_games @relay(mask: false)
      ...SearchContent_channels @relay(mask: false)
      ...SearchContent_videos @relay(mask: false)
      searchFor(userQuery: $userQuery, platform: $platform, target: $target)
        @skip(if: $noQuery) {
        relatedLiveChannels {
          ...SearchRelatedLiveChannels_channels
        }
      }
    }
  `,
});

export type SearchContentProps = {
  query: Search_QueryResponse;
  searchTerm: string;
  searchType: SearchType | undefined;
};

// istanbul ignore next: trivial
export const SearchContent: FC<SearchContentProps> = ({
  query,
  searchTerm,
  searchType,
}) => {
  switch (searchType) {
    case undefined:
      return <SearchOverview overview={query} searchTerm={searchTerm} />;
    case SearchType.Channel:
      return <InfiniteChannelList channels={query} searchTerm={searchTerm} />;
    case SearchType.Video:
      return <InfiniteVideoList searchTerm={searchTerm} videos={query} />;
    case SearchType.Category:
      return <InfiniteCategoryList games={query} searchTerm={searchTerm} />;
    default:
      return exhaustedCase(
        searchType,
        <SearchOverview overview={query} searchTerm={searchTerm} />,
      );
  }
};
SearchContent.displayName = 'SearchContent';
