import type { RelayRefetchProp } from 'react-relay/legacy';
import { SearchCategory, useDiscoveryTracking } from 'tachyon-discovery';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { InfiniteList } from '../../../common';
import { NoResultsFoundPage } from '../NoResultsFoundPage';
import { SearchCategoryCard } from './SearchCategoryCard';
import { SearchChannelCard } from './SearchChannelCard';
import { SearchRelatedLiveChannels } from './SearchRelatedLiveChannels';
import { SearchVideoCard } from './SearchVideoCard';
import {
  CategoryCard,
  ChannelCard,
  InfiniteCardListBase,
  SearchOverviewBase,
  VideoCard,
} from '.';

jest.mock('tachyon-discovery', () => ({
  ...jest.requireActual('tachyon-discovery'),
  useDiscoveryTracking: jest.fn(),
}));
const mockUseDiscoveryTracking = useDiscoveryTracking as jest.Mock;

describe(ChannelCard, () => {
  const setup = createShallowWrapperFactory(ChannelCard, () => ({
    channel: {
      ' $fragmentRefs': {
        SearchChannelCard_channel: true,
      },
      id: '1',
    },
    idx: 0,
    searchTerm: 'foo',
  }));

  it('wires up display tracking', () => {
    const onSearchResultImpression = jest.fn();
    const onSearchResultClick = jest.fn();
    mockUseDiscoveryTracking.mockImplementationOnce(() => ({
      onSearchResultClick,
      onSearchResultImpression,
    }));
    const { wrapper } = setup();

    const onDisplay = wrapper.find(SearchChannelCard).prop('onDisplay');
    const onClick = wrapper.find(SearchChannelCard).prop('onClick');

    onDisplay(true);
    expect(onSearchResultImpression).toHaveBeenCalledWith(
      expect.objectContaining({
        isLive: true,
        subSection: SearchCategory.Streams,
      }),
    );

    onClick(false);
    expect(onSearchResultClick).toHaveBeenCalledWith(
      expect.objectContaining({
        isLive: false,
        subSection: SearchCategory.Users,
      }),
    );

    onClick(true);
    expect(onSearchResultClick).toHaveBeenCalledWith(
      expect.objectContaining({
        isLive: true,
        subSection: SearchCategory.Streams,
      }),
    );
  });
});

describe(CategoryCard, () => {
  const setup = createShallowWrapperFactory(CategoryCard, () => ({
    game: {
      ' $fragmentRefs': {
        SearchCategoryCard_game: true,
      },
      id: '1',
    },
    idx: 0,
    searchTerm: 'foo',
  }));

  it('wires up display tracking', () => {
    const onSearchResultImpression = jest.fn();
    const onSearchResultClick = jest.fn();
    mockUseDiscoveryTracking.mockImplementationOnce(() => ({
      onSearchResultClick,
      onSearchResultImpression,
    }));
    const { wrapper } = setup();

    const onDisplay = wrapper.find(SearchCategoryCard).prop('onDisplay');
    const onClick = wrapper.find(SearchCategoryCard).prop('onClick');

    onDisplay();
    expect(onSearchResultImpression).toHaveBeenCalled();

    onClick();
    expect(onSearchResultClick).toHaveBeenCalled();
  });
});

describe(VideoCard, () => {
  const setup = createShallowWrapperFactory(VideoCard, () => ({
    idx: 0,
    searchTerm: 'foo',
    video: {
      ' $fragmentRefs': {
        SearchVideoCard_video: true,
      },
      id: '1',
    },
  }));

  it('wires up display tracking', () => {
    const onSearchResultImpression = jest.fn();
    const onSearchResultClick = jest.fn();
    mockUseDiscoveryTracking.mockImplementationOnce(() => ({
      onSearchResultClick,
      onSearchResultImpression,
    }));
    const { wrapper } = setup();

    const onDisplay = wrapper.find(SearchVideoCard).prop('onDisplay');
    const onClick = wrapper.find(SearchVideoCard).prop('onClick');

    onDisplay();
    expect(onSearchResultImpression).toHaveBeenCalled();

    onClick();
    expect(onSearchResultClick).toHaveBeenCalled();
  });
});

describe(InfiniteCardListBase, () => {
  const setup = createShallowWrapperFactory(InfiniteCardListBase, () => ({
    content: [] as any,
    itemRenderer: jest.fn() as any,
    relay: { refetch: jest.fn() } as unknown as RelayRefetchProp,
    searchType: 'CHANNEL',
  }));

  it('renders the "InfiniteList" when there is content', () => {
    const { wrapper } = setup({
      content: { cursor: 'foo', items: [{ id: '1' }, { id: '2' }] },
    });
    expect(wrapper.find(InfiniteList)).toExist();
  });

  it('renders the "NoResultsFoundPage" when there is not content', () => {
    const { wrapper } = setup();
    expect(wrapper.find(NoResultsFoundPage)).toExist();
  });
});

describe(SearchOverviewBase, () => {
  const setup = createShallowWrapperFactory(SearchOverviewBase, () => ({
    overview: {
      ' $refType': 'SearchContent_overview',
      searchFor: {
        channels: null,
        games: null,
        relatedLiveChannels: null,
        videos: null,
      },
    },
    searchTerm: 'foo',
  }));

  it('renders the "NoResultsFoundPage" when there is not content', () => {
    const { wrapper } = setup();
    expect(wrapper.find(NoResultsFoundPage)).toExist();
  });

  it('renders channels when present', () => {
    const { wrapper } = setup({
      overview: {
        searchFor: {
          channels: {
            cursor: 'foo',
            items: [{ id: '1' }, { id: '2' }],
          },
        },
      },
    });
    expect(wrapper.find(ChannelCard)).toExist();
  });

  it('renders related live channels when present', () => {
    const { wrapper } = setup({
      overview: {
        searchFor: {
          relatedLiveChannels: {
            ' $fragmentRefs': { SearchRelatedLiveChannels_channels: true },
          },
        },
      },
    });
    expect(wrapper.find(SearchRelatedLiveChannels)).toExist();
  });

  it('renders games when present', () => {
    const { wrapper } = setup({
      overview: {
        searchFor: {
          games: {
            cursor: 'foo',
            items: [{ id: '1' }, { id: '2' }],
          },
        },
      },
    });
    expect(wrapper.find(CategoryCard)).toExist();
  });

  it('renders videos when present', () => {
    const { wrapper } = setup({
      overview: {
        searchFor: {
          videos: {
            cursor: 'foo',
            items: [{ id: '1' }, { id: '2' }],
          },
        },
      },
    });
    expect(wrapper.find(VideoCard)).toExist();
  });
});
