import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import {
  formatFollowers,
  formatLastLive,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AlignItems,
  Avatar,
  Color,
  CoreText,
  Display,
  JustifyContent,
  Layout,
} from 'twitch-core-ui';
import { SearchContentChannelName } from '../SearchContentChannelName';
import { ScImageContainer } from '../utils';
import type { SearchOfflineChannel_channel } from './__generated__/SearchOfflineChannel_channel.graphql';

export interface SearchOfflineChannelProps {
  channel: SearchOfflineChannel_channel;
  isPartner?: boolean | null;
}

export const SearchOfflineChannelBase: FC<SearchOfflineChannelProps> = ({
  channel,
  isPartner,
}) => {
  const intl = useIntl();
  const followerCount = channel.followers?.totalCount ?? null;

  return (
    <Layout alignItems={AlignItems.Center} display={Display.Flex}>
      <ScImageContainer
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
      >
        <Avatar
          alt=""
          size={80}
          userLogin={channel.login}
          {...getChannelThumbnailSrcAndSrcSet(channel.profileImageURL)}
        />
      </ScImageContainer>
      <Layout padding={{ left: 1 }}>
        <SearchContentChannelName
          displayName={getFormattedUserDisplayName(channel)}
          isPartner={isPartner}
        />
        {channel.lastBroadcast?.startedAt && (
          <CoreText color={Color.Alt2}>
            {formatLastLive(intl, channel.lastBroadcast.startedAt)}
          </CoreText>
        )}
        {followerCount !== null && (
          <CoreText color={Color.Alt2}>
            {formatFollowers(intl, followerCount)}
          </CoreText>
        )}
      </Layout>
    </Layout>
  );
};

SearchOfflineChannelBase.displayName = 'SearchOfflineChannelBase';

export const SearchOfflineChannel = createFragmentContainer(
  SearchOfflineChannelBase,
  {
    channel: graphql`
      fragment SearchOfflineChannel_channel on User {
        displayName
        followers {
          totalCount
        }
        lastBroadcast {
          startedAt
        }
        login
        profileImageURL(width: 150)
      }
    `,
  },
);
