import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import {
  StreamType,
  StreamTypeIndicator,
  formatLiveViewersCount,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AlignItems,
  CardImage,
  CardImageSize,
  Color,
  CoreText,
  Display,
  Layout,
  Overflow,
  Position,
  TextTransform,
} from 'twitch-core-ui';
import { SearchContentChannelName } from '../SearchContentChannelName';
import { ScImageContainer } from '../utils';
import type { SearchStreamCard_stream } from './__generated__/SearchStreamCard_stream.graphql';

export interface SearchStreamCardProps {
  isPartner?: boolean | null;
  stream: SearchStreamCard_stream;
}

export const SearchStreamCardBase: FC<SearchStreamCardProps> = ({
  isPartner,
  stream,
}) => {
  const { formatMessage } = useIntl();

  return (
    <Layout alignItems={AlignItems.Center} display={Display.Flex}>
      <ScImageContainer position={Position.Relative}>
        <CardImage
          size={CardImageSize.Size16}
          {...getChannelThumbnailSrcAndSrcSet(stream.previewImageURL)}
          alt=""
        />
        <Layout attachLeft attachTop margin={0.5} position={Position.Absolute}>
          <StreamTypeIndicator
            isOverlay
            isPulsing={false}
            streamType={StreamType.Live}
          />
        </Layout>
      </ScImageContainer>
      <Layout overflow={Overflow.Hidden} padding={{ left: 1 }}>
        {stream.broadcaster && (
          <SearchContentChannelName
            displayName={getFormattedUserDisplayName(stream.broadcaster)}
            isPartner={isPartner}
          />
        )}
        <CoreText color={Color.Alt2} transform={TextTransform.Capitalize}>
          {stream.game?.displayName}
        </CoreText>
        {stream.viewersCount !== null && (
          <CoreText color={Color.Alt2}>
            {formatLiveViewersCount(formatMessage, stream.viewersCount)}
          </CoreText>
        )}
        <CoreText color={Color.Alt2} ellipsis>
          {stream.broadcaster?.broadcastSettings?.title}
        </CoreText>
      </Layout>
    </Layout>
  );
};

SearchStreamCardBase.displayName = 'SearchStreamCardBase';

export const SearchStreamCard = createFragmentContainer(SearchStreamCardBase, {
  stream: graphql`
    fragment SearchStreamCard_stream on Stream {
      broadcaster {
        displayName
        login
        broadcastSettings {
          title
        }
      }
      game {
        displayName
        name
      }
      id
      previewImageURL
      viewersCount
    }
  `,
});
