import { useState } from 'react';
import { graphql } from 'react-relay/hooks';
import styled from 'styled-components';
import { SearchQueryType, useDiscoveryTracking } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { safeAreaHorizontalPaddingToRemoveInsets } from 'tachyon-more-ui';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { Enum, exhaustedCase, flattenHeaderOrParam } from 'tachyon-utils';
import {
  Background,
  ButtonIcon,
  ButtonIconSize,
  Color,
  Display,
  Input,
  InputSize,
  InputType,
  JustifyContent,
  Layout,
  Position,
  SVGAsset,
  Tabs,
  ZIndex,
} from 'twitch-core-ui';
import {
  SEARCH_QUERY_PLATFORM,
  SEARCH_TERM_QUERY_PARAM_KEY,
  SEARCH_TYPE_QUERY_PARAM_KEY,
} from '../../../config';
import { RouteName, useTachyonRouter } from '../../../routing';
import { pageHeadQueryVariables } from '../../growth';
import { Page } from '../../layouts';
import type { PageWithQueryProps, TomorrowPage } from '../types';
import { SearchAutocomplete } from './SearchAutocomplete';
import { SearchContent } from './SearchContent';
import type {
  Search_QueryResponse,
  Search_QueryVariables,
} from './__generated__/Search_Query.graphql';
import { SearchType, getSearchLink } from './utils';

type SearchInitialProps = {
  queryVariables: Search_QueryVariables;
};

export type SearchProps = PageWithQueryProps<
  Search_QueryResponse,
  SearchInitialProps
>;

// istanbul ignore next: trivial
const ScSearchBarContainer = styled(Layout)`
  ${safeAreaHorizontalPaddingToRemoveInsets}
`;

// istanbul ignore next: trivial
const ScSearchContent = styled(Layout)`
  height: 100%;
  max-width: 1200px;
  overflow: auto;
`;

// istanbul ignore next: trivial
const ScSearchInput = styled(Layout)`
  ${safeAreaHorizontalPaddingToRemoveInsets}
`;

export const Search: TomorrowPage<SearchInitialProps, SearchProps> = ({
  loading,
  queryVariables,
  ...props
}) => {
  const { formatMessage } = useIntl();
  const router = useTachyonRouter();
  const { onSearchInputFocus, onSearchQuerySubmit } = useDiscoveryTracking();
  const { currentQuery } = useRouterUtils();
  const searchTerm = queryVariables.userQuery;
  const [showAutocomplete, setShowAutocomplete] = useState(!searchTerm);
  const [autocompleteSearch, setAutocompleteSearch] = useState(searchTerm);
  const searchType = Enum.convertValueFromExternal(
    SearchType,
    flattenHeaderOrParam(currentQuery[SEARCH_TYPE_QUERY_PARAM_KEY]),
  );

  let activeTabIndex = 0;
  // istanbul ignore next: trivial
  switch (searchType) {
    case undefined:
      activeTabIndex = 0;
      break;
    case SearchType.Channel:
      activeTabIndex = 1;
      break;
    case SearchType.Category:
      activeTabIndex = 2;
      break;
    case SearchType.Video:
      activeTabIndex = 3;
      break;
    default:
      exhaustedCase(searchType, 0);
  }

  return (
    <>
      <ScSearchBarContainer
        as="nav"
        attachTop
        background={Background.Base}
        borderBottom={!showAutocomplete}
        fullWidth
        position={Position.Fixed}
        zIndex={ZIndex.Above}
      >
        <Layout padding={{ left: 1, right: 2, top: 1 }}>
          <ScSearchInput display={Display.Flex} fullWidth>
            <Layout margin={{ right: 1 }}>
              <ButtonIcon
                aria-label={formatMessage('Back', 'SearchInput')}
                icon={SVGAsset.AngleLeft}
                onClick={router.back}
                size={ButtonIconSize.Large}
              />
            </Layout>
            <Layout flexGrow={1}>
              <Input
                onChange={(e) => {
                  setAutocompleteSearch(e.currentTarget.value);
                }}
                onClick={() => {
                  setShowAutocomplete(true);
                }}
                onFocus={onSearchInputFocus}
                onKeyDown={(e) => {
                  if (e.key === 'Enter' && autocompleteSearch) {
                    e.currentTarget.blur();
                    router.replace(
                      { route: RouteName.Search },
                      {
                        [SEARCH_TERM_QUERY_PARAM_KEY]: autocompleteSearch,
                      },
                    );
                    onSearchQuerySubmit({
                      query: autocompleteSearch,
                      type: SearchQueryType.UserTyped,
                    });
                    setShowAutocomplete(false);
                  }
                }}
                placeholder={formatMessage('Search...', 'SearchInput')}
                refDelegate={(ref) => {
                  if (showAutocomplete) {
                    ref?.focus();
                  }
                }}
                size={InputSize.Large}
                type={InputType.Search}
                value={showAutocomplete ? autocompleteSearch : searchTerm}
              />
            </Layout>
          </ScSearchInput>
          {!showAutocomplete && (
            <Layout padding={{ left: 1, top: 1 }}>
              <Tabs
                activeTabIndex={activeTabIndex}
                borderBottom={false}
                justifyContent={JustifyContent.Start}
                tabs={[
                  {
                    label: formatMessage('Top', 'SearchResults'),
                    ...getSearchLink(searchTerm),
                  },
                  {
                    label: formatMessage('Channels', 'SearchResults'),
                    ...getSearchLink(searchTerm, SearchType.Channel),
                  },
                  {
                    label: formatMessage('Categories', 'SearchResults'),
                    ...getSearchLink(searchTerm, SearchType.Category),
                  },
                  {
                    label: formatMessage('Videos', 'SearchResults'),
                    ...getSearchLink(searchTerm, SearchType.Video),
                  },
                ]}
              />
            </Layout>
          )}
        </Layout>
      </ScSearchBarContainer>
      <Page hideTopNav loading={loading}>
        <Layout
          background={showAutocomplete ? Background.Base : Background.Inherit}
          fullHeight
          fullWidth
          margin={{ top: showAutocomplete ? 0 : 5 }}
          padding={{ top: 5 }}
        >
          <ScSearchContent
            breakpointMedium={{
              padding: { x: 5 },
            }}
            breakpointSmall={{
              padding: { x: 3 },
            }}
            color={Color.Base}
            flexGrow={1}
            padding={{ x: 1 }}
          >
            {showAutocomplete ? (
              <SearchAutocomplete
                hide={() => {
                  setShowAutocomplete(false);
                  setAutocompleteSearch('');
                }}
                searchTerm={autocompleteSearch}
                suggestions={props}
              />
            ) : (
              <SearchContent
                query={props}
                searchTerm={searchTerm}
                searchType={searchType}
              />
            )}
          </ScSearchContent>
        </Layout>
      </Page>
    </>
  );
};
Search.displayName = 'Search';

Search.pageviewTracking = defaultPageviewTracking;
Search.requiresJsForInteractivity = false;
Search.handlesLoading = true;

// istanbul ignore next: trivial
Search.getInitialProps = (ctx) => {
  const searchTerm =
    flattenHeaderOrParam(ctx.query[SEARCH_TERM_QUERY_PARAM_KEY]) ?? '';

  return {
    queryVariables: {
      ...pageHeadQueryVariables(ctx),
      noQuery: !searchTerm,
      noQueryFragment: true,
      platform: SEARCH_QUERY_PLATFORM,
      queryFragment: '',
      target: null,
      userQuery: searchTerm,
    },
  };
};

Search.query = graphql`
  query Search_Query(
    $noQuery: Boolean!
    $noQueryFragment: Boolean!
    $platform: String!
    $queryFragment: String!
    $requestID: ID
    $target: SearchForTarget
    $userQuery: String!
    $url: String!
  ) {
    ...SearchAutocomplete_suggestions
      @arguments(
        queryFragment: $queryFragment
        requestID: $requestID
        noQueryFragment: $noQueryFragment
      )
    ...SearchContent_channels
    ...SearchContent_games
    ...SearchContent_overview
    ...SearchContent_videos
    ...PageHead_query
  }
`;
