import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { useConst } from 'tachyon-utils';
import {
  ButtonSize,
  Display,
  FlexDirection,
  Layout,
  Overflow,
} from 'twitch-core-ui';
import {
  ContentInfoBox,
  VideoTagList,
  WEB_SHARE_BUTTON_HEIGHT,
  WebShareButton,
  canShare,
} from '../../../common';
import { BranchEvent, TachyonBranchTimedTracker } from '../../../growth';
import { VodList } from '../VodList';
import type { VodPanel_video } from './__generated__/VodPanel_video.graphql';

export type VodPanelProps = {
  video: VodPanel_video;
};

export const VodPanelBase: FC<VodPanelProps> = ({ video }) => {
  const sharingSupported = useConst(canShare);
  const { formatMessage } = useIntl();
  const { game, owner, title } = video;

  return (
    <Layout
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      fullHeight
      overflow={Overflow.Hidden}
    >
      <ContentInfoBox
        contentMeta={{
          description: title,
          header: owner?.displayName ?? null,
        }}
        owner={owner}
      >
        {game && (
          <VideoTagList
            gameAlias={game.name}
            page={TagPage.Stream}
            video={video}
          />
        )}
      </ContentInfoBox>
      {sharingSupported && (
        <Layout height={WEB_SHARE_BUTTON_HEIGHT} padding={1}>
          <WebShareButton fullWidth size={ButtonSize.Large}>
            {formatMessage('Share this video', 'DiscoverMore')}
          </WebShareButton>
        </Layout>
      )}
      {owner?.videos && <VodList videos={owner.videos} />}
      <TachyonBranchTimedTracker
        delaySecondsUntilTrack={5 * 60}
        event={BranchEvent.FiveMinutePlay}
      />
    </Layout>
  );
};

VodPanelBase.displayName = 'VodPanelBase';

export const VodPanel = createFragmentContainer(VodPanelBase, {
  video: graphql`
    fragment VodPanel_video on Video {
      owner {
        login
        displayName
        profileImageURL(width: 50)
        videos(sort: TIME, first: 8) {
          ...VodList_videos
        }
      }
      title
      game {
        name
        displayName
      }
      ...TagList_video
    }
  `,
});
