import type { FC } from 'react';
import { Pageview } from 'tachyon-event-tracker';
import { LatencyTransitionComplete } from 'tachyon-latency-tracker';
import type {
  TachyonPageInitialProps,
  TachyonPageProps,
  TachyonPageType,
} from 'tachyon-next-types';
import { GraphQLErrorType } from 'tachyon-relay';
import type { TomorrowPageExtensions } from '../../../../config';
import { PageHead } from '../../../growth';
// eslint-disable-next-line import/no-internal-modules
import type { PageHead_query$key } from '../../../growth/PageHead/__generated__/PageHead_query.graphql';
import { NetworkNotification, Page as PageLayout } from '../../../layouts';
import { AppShell } from '../../AppShell';
import { DefaultOfflinePage } from './DefaultOfflinePage';

type PageRendererProps = {
  Page: TachyonPageType<
    TachyonPageInitialProps,
    TachyonPageProps,
    TomorrowPageExtensions<TachyonPageProps>
  >;
  error: Error | null;
  pageProps: TachyonPageProps;
  relayQueryResponse: unknown;
};

export const PageRenderer: FC<PageRendererProps> = ({
  Page,
  error,
  pageProps,
  relayQueryResponse,
}) => {
  let isOffline = false;
  if (error) {
    if (error.message === GraphQLErrorType.ConnectionError) {
      isOffline = true;
    } else {
      throw error;
    }
  }

  const loading = !!Page.query && !isOffline && !relayQueryResponse;
  const componentProps = {
    ...pageProps,
    ...(relayQueryResponse as Record<string, unknown> | null | undefined),
    loading,
  };

  return (
    <>
      {!loading && (
        <>
          <Pageview
            {...Page.pageviewTracking(componentProps)}
            clientOffline={isOffline}
          />

          <LatencyTransitionComplete
            requiresJsForInteractivity={Page.requiresJsForInteractivity}
          />
        </>
      )}
      <NetworkNotification
        enable={!loading && Page !== AppShell}
        isOfflinePage={isOffline}
      >
        <PageHead
          isOffline={isOffline}
          loading={loading}
          query={relayQueryResponse as PageHead_query$key | undefined}
        >
          {isOffline ? (
            Page.OfflinePage ? (
              <Page.OfflinePage {...componentProps} />
            ) : (
              <DefaultOfflinePage />
            )
          ) : loading && !Page.handlesLoading ? (
            <PageLayout loading={loading} />
          ) : (
            <Page {...componentProps} />
          )}
        </PageHead>
      </NetworkNotification>
    </>
  );
};

PageRenderer.displayName = 'PageRenderer';
