import type { GraphQLTaggedNode } from 'react-relay/hooks';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { GraphQLErrorType } from 'tachyon-relay';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { NetworkNotification, Page } from '../../../layouts';
import { AppShell } from '../../AppShell';
import type { TomorrowPage } from '../../types';
import { DefaultOfflinePage } from './DefaultOfflinePage';
import { PageRenderer } from '.';

jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  isOffline: jest.fn(() => false),
}));

function mockPage(): TomorrowPage<{}, {}> {
  const page = () => <div />;
  page.pageviewTracking = defaultPageviewTracking;
  page.displayName = 'TestPage';
  page.requiresJsForInteractivity = false;
  return page;
}

describe(PageRenderer, () => {
  const setup = createShallowWrapperFactory(PageRenderer, () => ({
    Page: mockPage(),
    error: null,
    pageProps: {},
    relayQueryResponse: {},
  }));

  describe('loading', () => {
    it('renders the page content when the page does not have a query', () => {
      const TestPage = mockPage();
      const { wrapper } = setup({ Page: TestPage });
      expect(wrapper.find(TestPage)).toExist();
    });

    it('renders the loading experience when the query response is null and the page does not opt in to handlesLoading', () => {
      const TestPage = mockPage();
      TestPage.query = {} as GraphQLTaggedNode;
      const { wrapper } = setup({ Page: TestPage, relayQueryResponse: null });
      expect(wrapper.find(TestPage)).not.toExist();
      expect(wrapper.find(Page)).toExist();
      expect(wrapper.find(Page)).toHaveProp({ loading: true });
    });

    it('renders the page experience when the query response is null and the page opts in to handlesLoading', () => {
      const TestPage = mockPage();
      TestPage.query = {} as GraphQLTaggedNode;
      TestPage.handlesLoading = true;
      const { wrapper } = setup({ Page: TestPage, relayQueryResponse: null });
      expect(wrapper.find(TestPage)).toExist();
      expect(wrapper.find(TestPage)).toHaveProp({ loading: true });
      expect(wrapper.find(Page)).not.toExist();
    });

    it('renders the page content when the query response is present', () => {
      const TestPage = mockPage();
      TestPage.query = {} as GraphQLTaggedNode;

      const { wrapper } = setup({ Page: TestPage, relayQueryResponse: {} });
      expect(wrapper.find(TestPage)).toExist();
    });

    it('throws when there is an error (not connection-error) fetching the GQL request', () => {
      expect(() =>
        setup({
          error: Error(GraphQLErrorType.UndefinedResponse),
        }),
      ).toThrow(GraphQLErrorType.UndefinedResponse);
    });

    describe('renders the offline experience', () => {
      it('when there is a network error (connection-error) fetching the GQL request', () => {
        const { wrapper } = setup({
          error: Error(GraphQLErrorType.ConnectionError),
        });
        expect(wrapper.find(DefaultOfflinePage)).toExist();
      });

      describe('with', () => {
        it("the page's OfflinePage when present", () => {
          const MockPageWithOffline: TomorrowPage<{}, {}> = () => <div />;
          MockPageWithOffline.pageviewTracking = defaultPageviewTracking;
          MockPageWithOffline.displayName = 'TestPage';
          MockPageWithOffline.requiresJsForInteractivity = false;
          MockPageWithOffline.OfflinePage = () => <div id="offline-page" />;

          const { wrapper } = setup({
            Page: MockPageWithOffline,
            error: Error(GraphQLErrorType.ConnectionError),
          });

          expect(wrapper.find(MockPageWithOffline.OfflinePage)).toExist();
          expect(wrapper.find(DefaultOfflinePage)).not.toExist();
        });

        it('the fallback if the page does not have an OfflinePage', () => {
          const { wrapper } = setup({
            error: Error(GraphQLErrorType.ConnectionError),
          });
          expect(wrapper.find(DefaultOfflinePage)).toExist();
        });
      });
    });
  });

  describe('NetworkNotification', () => {
    it('is enabled when not loading', () => {
      const { wrapper } = setup({ relayQueryResponse: {} });
      expect(wrapper.find(NetworkNotification)).toHaveProp({ enable: true });
    });

    it('is not enabled when the page is the AppShell', () => {
      const { wrapper } = setup({ Page: AppShell, relayQueryResponse: {} });
      expect(wrapper.find(NetworkNotification)).toHaveProp({ enable: false });
    });
  });
});
