import type { FC } from 'react';
import { useStaticEnvironment } from 'tachyon-environment';
import { useCustomLatency } from 'tachyon-latency-tracker';
import { logger } from 'tachyon-logger';
import { useEffectOnce } from 'tachyon-utils';
import { Workbox } from 'workbox-window';

// This needs to be the same for every build of the app, so that the SW can
// update properly over time
const SERVICE_WORKER_HREF = '/service-worker.js';

/**
 * Manages feature detection and registration of our Service Worker which begins
 * once the app has finished loading.
 */
export const ServiceWorker: FC = () => {
  const {
    client: { agentInfo },
  } = useStaticEnvironment();
  const reportLatencyEvent = useCustomLatency();

  useEffectOnce(() => {
    if (agentInfo.satisfies({ googlebot: '>0' })) {
      return;
    }

    // users must have promises for the usages of the SW API that workbox uses
    if (typeof Promise === 'undefined' || !('serviceWorker' in navigator)) {
      logger.info({
        category: 'ServiceWorker',
        message: 'SW not supported by browser',
        package: 'tomorrow',
      });

      return;
    }

    // Reports when the service worker is installed in browser.
    const onInstall = reportLatencyEvent('workerInstalled');
    // Reports when the service worker is waiting in browser.
    const onWaiting = reportLatencyEvent('workerWaiting');
    // Reports when the service worker is activated in browser.
    const onActivated = reportLatencyEvent('workerActivated');
    // Reports when the service worker is controlling browser.
    const onControlling = reportLatencyEvent('workerControlling');

    const workbox = new Workbox(SERVICE_WORKER_HREF);
    workbox.addEventListener('installed', onInstall);
    workbox.addEventListener('waiting', onWaiting);
    workbox.addEventListener('activated', onActivated);
    workbox.addEventListener('controlling', onControlling);

    // By default this method delays registration until after the window has loaded
    workbox.register().catch((error: Error) => {
      logger.error({
        category: 'ServiceWorker',
        error,
        message: 'SW registration failed',
        package: 'tomorrow',
      });
    });

    return () => {
      workbox.removeEventListener('installed', onInstall);
      workbox.removeEventListener('waiting', onWaiting);
      workbox.removeEventListener('activated', onActivated);
      workbox.removeEventListener('controlling', onControlling);
    };
  });

  return null;
};

ServiceWorker.displayName = 'ServiceWorker';
