import { getOtherEnvMock, useStaticEnvironment } from 'tachyon-environment';
import { useCustomLatency } from 'tachyon-latency-tracker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { Workbox } from 'workbox-window';
import { ServiceWorker } from '.';

jest.mock('workbox-window');
const mockWorkbox = Workbox as jest.Mock<Workbox>;
jest.mock('tachyon-logger', () => ({ logger: { info: jest.fn() } }));

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: jest.fn(),
}));
const mockUseStaticEnvironment = useStaticEnvironment as jest.Mock;
mockUseStaticEnvironment.mockImplementation(() => getOtherEnvMock());

jest.mock('tachyon-latency-tracker', () => ({
  ...jest.requireActual('tachyon-latency-tracker'),
  useCustomLatency: jest.fn(),
}));
(useCustomLatency as jest.Mock).mockImplementation(() => () => () => jest.fn());

const setup = createMountWrapperFactory(ServiceWorker);

describe(ServiceWorker, () => {
  beforeEach(() => {
    // Create a mockable copy of Workbox in order to access its specific functions
    // Eventually should replace this using the sw-mock (https://jira.twitch.com/browse/MWC-1773)
    mockWorkbox.mockReset();
    (window.navigator as any).serviceWorker = {
      getRegistrations: () => Promise.resolve([]),
    };
  });

  afterEach(() => {
    delete (window.navigator as any).serviceWorker;
  });

  describe('when the browser does not support Service Workers', () => {
    it('skips registration when API is not available', () => {
      delete (window.navigator as any).serviceWorker;
      setup();

      const wbInstance = mockWorkbox.mock.instances[0];
      // represents `new Workbox()` not being called
      expect(wbInstance).toBeUndefined();
    });

    it('skips registration when user is googlebot', () => {
      mockUseStaticEnvironment.mockImplementationOnce(() =>
        getOtherEnvMock({
          ua: 'Mozilla/5.0 AppleWebKit/537.36 (KHTML, like Gecko; compatible; Googlebot/2.1; +http://www.google.com/bot.html) Chrome/80.0.3987.92 Safari/537.36',
        }),
      );
      setup();

      const wbInstance = mockWorkbox.mock.instances[0];
      // represents `new Workbox()` not being called
      expect(wbInstance).toBeUndefined();
    });

    it('skips registration when user is googlebot mobile', () => {
      mockUseStaticEnvironment.mockImplementationOnce(() =>
        getOtherEnvMock({
          ua: 'Mozilla/5.0 (Linux; Android 6.0.1; Nexus 5X Build/MMB29P) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/80.0.3987.92 Mobile Safari/537.36 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)',
        }),
      );
      setup();

      const wbInstance = mockWorkbox.mock.instances[0];
      // represents `new Workbox()` not being called
      expect(wbInstance).toBeUndefined();
    });
  });

  describe('when the browser supports Service Workers', () => {
    it('registers the SW', () => {
      const mockRegister = jest.fn(() => ({ catch: jest.fn() }));
      mockWorkbox.mockImplementation(
        () =>
          ({
            addEventListener: jest.fn(),
            register: mockRegister,
            removeEventListener: jest.fn(),
          } as unknown as Workbox),
      );
      setup();
      expect(mockRegister).toHaveBeenCalledTimes(1);
    });
  });
});
