import { configureClientLogging, logger } from 'tachyon-logger';
import type {
  ConfigureTachyonRelayOpts,
  TwitchPayloadError,
} from 'tachyon-relay';
import { configureTachyonRelay } from 'tachyon-relay';
import {
  getCurrentTwitchDomain,
  isBrowser,
  setCookieDomain,
} from 'tachyon-utils';
import {
  ALLOWED_PARTIAL_RESPONSE_QUERIES,
  CLIENT_ID,
  DEFAULT_PLATFORM,
  FAIL_SERVICES,
  GQL_DEBUG_MODE,
  GQL_SANDBOX_ENDPOINT,
  IGNORED_ERROR_MESSAGES,
  THROTTLED_ERROR_MESSAGES,
  THROTTLED_ERROR_MESSAGE_THRESHOLD,
} from '../../../../config';

// istanbul ignore next: mostly testing mocks
export function configure(): void {
  const relayOpts: ConfigureTachyonRelayOpts = {
    clientId: CLIENT_ID,
    errorsArrayIsFatal,
  };

  if (process.env.NODE_ENV !== 'production' && GQL_DEBUG_MODE) {
    relayOpts.debug = GQL_DEBUG_MODE;
    relayOpts.gqlEndpoint = GQL_SANDBOX_ENDPOINT;
  }

  if (isBrowser()) {
    const buildId = process.env.BUILD_ID;
    // reach into next data since we're outside of the react tree on the client
    const { appEnvironment, language } = window.__NEXT_DATA__.props;

    logger.debug({
      category: 'configure',
      context: {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        appEnvironment,
        buildId,
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        language,
      },
      message: 'Configuring application',
      package: 'tomorrow',
    });

    if (process.env.NODE_ENV === 'production') {
      // configure error reporting (this should happen first/asap)
      try {
        configureClientLogging({
          // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
          appEnvironment,
          appSentryDsn: process.env.SENTRY_DSN,
          buildId,
          ignoreErrors: IGNORED_ERROR_MESSAGES,
          // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
          language,
          platform: DEFAULT_PLATFORM,
          throttledErrorMessageThreshold: THROTTLED_ERROR_MESSAGE_THRESHOLD,
          throttledErrorMessages: THROTTLED_ERROR_MESSAGES,
        });
      } catch {
        // If this fails still try to start the app, don't bother logging though since our
        // logger isn't shipping logs.
      }
    }

    // configure cookie domain
    try {
      const domain = getCurrentTwitchDomain(window.location.hostname);
      if (domain) {
        setCookieDomain(domain);
        // TODO: audit necessity
        // https://jira.twitch.com/browse/MWC-2473
        // We need this for certain player cross-frame communication.
        // It makes sense to put this here instead of the player component
        // since it is a fairly global change.
        document.domain = domain;
        logger.debug({
          category: 'configure',
          message: `Setting the domain for cookies and the document to ${domain}`,
          package: 'tomorrow',
        });
      }
    } catch {
      // This can throw a security exception if the page isn't on a twitch.tv
      // subdomain. If it does then there isn't really much we can do about
      // this. If we didn't have staging domains off of twitch.tv then we
      // could consider this an _actual_ security error.
      logger.warn({
        category: 'configure',
        message: `Failed to set domain for ${window.location.hostname}`,
        package: 'tomorrow',
      });
    }

    // by default we only simulate failure in the browser to enable a better
    // debugging experience
    if (
      process.env.NODE_ENV !== 'production' &&
      GQL_DEBUG_MODE &&
      FAIL_SERVICES.length
    ) {
      relayOpts.failServices = FAIL_SERVICES;
    }
  }

  // not using isBrowser() to ensure DCE and thus no `require('https')` in the browser
  if (typeof window === 'undefined') {
    // eslint-disable-next-line @typescript-eslint/no-var-requires, @typescript-eslint/no-unsafe-assignment
    const { Agent } = require('https');
    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-call
    relayOpts.serverHttpsAgent = new Agent({
      keepAlive: true,
    });
  }

  configureTachyonRelay(relayOpts);
}

/**
 * Callback passed to tachyon-relay to enable partial response handling of
 * GraphQL responses (as opposed to the normal guidance of treating the entire
 * payload as an error). It receives the errors array and the raw query sent to
 * the server in order to make decisions.
 *
 * When handling partial responses, it is extremely important to handle errors
 * at all service boundaries. The GraphQL types will help with this, but care
 * must be taken to render appropriate fallback content for missing data.
 *
 * To opt specific queries in to partial response handling, update
 * `ALLOWED_PARTIAL_RESPONSE_QUERIES` in `tachyon-lib/config/gql`. Use the exact
 * name of your query, as the relay-compiler ensures name uniqueness and will
 * help prevent collisions.
 *
 * Note that this does not do anything with the errors array and blindly gates
 * solely on the query name. This behavior should probably evolve.
 */
export function errorsArrayIsFatal(
  _errors: TwitchPayloadError[],
  queryName: string,
): boolean {
  if (ALLOWED_PARTIAL_RESPONSE_QUERIES.includes(queryName)) {
    return false;
  }

  return true;
}
