import type { FC } from 'react';
import type { SpadeDiscoveryEvent } from 'tachyon-discovery';
import { DiscoveryTrackingRoot, PageviewMedium } from 'tachyon-discovery';
import { useCustomTracking } from 'tachyon-event-tracker';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { RouteName, routeNameFromPathname } from '../../../../../routing';

const SEARCH_CORRELATION_ROUTES = [
  RouteName.Search,
  RouteName.Channel,
  RouteName.Vod,
];

/**
 * A map of which mediums should be attributed to what routes.
 */
const ROUTE_TO_MEDIUM_MAP: Partial<Record<string, PageviewMedium>> = {
  [RouteName.GamesDirectory]: PageviewMedium.Browse,
  [RouteName.Homepage]: PageviewMedium.Discover,
  [RouteName.Search]: PageviewMedium.Search,
  [RouteName.ChannelHome]: PageviewMedium.ChannelHome,
  [RouteName.ChannelAbout]: PageviewMedium.ChannelAbout,
  [RouteName.ChannelClips]: PageviewMedium.ChannelClips,
  [RouteName.ChannelVideos]: PageviewMedium.ChannelVideos,
  [RouteName.ChannelSchedule]: PageviewMedium.ChannelSchedule,
};

// istanbul ignore next: trivial
const getMediumFromRoute = (routeName: string): PageviewMedium | undefined => {
  return ROUTE_TO_MEDIUM_MAP[routeName];
};

// istanbul ignore next: trivial
export const TomorrowDiscoveryTrackingRoot: FC = ({ children }) => {
  const trackEvent = useCustomTracking<SpadeDiscoveryEvent>();
  const { currentPathname } = useRouterUtils();

  return (
    <DiscoveryTrackingRoot
      children={children}
      currentRouteName={routeNameFromPathname(currentPathname)}
      getMediumFromRoute={getMediumFromRoute}
      onEvent={trackEvent}
      searchCorrelationRoutes={SEARCH_CORRELATION_ROUTES}
    />
  );
};

TomorrowDiscoveryTrackingRoot.displayName = 'TomorrowDiscoveryTrackingRoot';
