import type { FC } from 'react';
import { useCallback } from 'react';
import { useCurrentUser } from 'tachyon-auth';
import { useStaticEnvironment } from 'tachyon-environment';
import { useReportEvent } from 'tachyon-event-reporter';
import { EventTrackerRoot } from 'tachyon-event-tracker';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { flattenHeaderOrParam } from 'tachyon-utils';
import { routeNameFromPathname } from '../../../../../routing';
import { getTwilightUserCookie } from '../../../../../utils';

type TomorrowEventTrackerRootProps = {
  inError: boolean;
};

// istanbul ignore next: trivial
export const TomorrowEventTrackerRoot: FC<TomorrowEventTrackerRootProps> = ({
  children,
  inError,
}) => {
  const reportEvent = useReportEvent();
  const {
    currentPathname,
    currentQuery: { tt_content, tt_email_id, tt_medium },
  } = useRouterUtils();
  const { client, common } = useStaticEnvironment();
  const currentRouteName = routeNameFromPathname(currentPathname);
  const { getUserTracking, loggedIn } = useCurrentUser();

  const getTmwUserTracking = useCallback(
    () =>
      getUserTracking() ??
      // In local development we piggy back on Twilight's login for DX
      // Remove when TMW has a login UI.
      (common.appEnvironment === 'development' && loggedIn
        ? getTwilightUserCookie()
        : undefined),
    [common.appEnvironment, loggedIn, getUserTracking],
  );

  return (
    <EventTrackerRoot
      children={children}
      getUserTracking={getTmwUserTracking}
      interactionMedium={currentRouteName}
      location={inError ? 'internal_error' : currentRouteName}
      onEvent={reportEvent}
      staticProperties={{
        appVersion: common.appVersion,
        clientApp: common.clientApp,
        deviceID: client.deviceID,
        language: common.language,
        netInfo: client.netInfo,
        platform: common.platform,
        sessionID: client.sessionID,
      }}
      twitchTracking={{
        content: flattenHeaderOrParam(tt_content),
        emailID: flattenHeaderOrParam(tt_email_id),
        medium: flattenHeaderOrParam(tt_medium),
      }}
    />
  );
};

TomorrowEventTrackerRoot.displayName = 'TomorrowEventTrackerRoot';
