import type { FC, MutableRefObject } from 'react';
import { useEffect, useRef } from 'react';
import { useRelayEnvironment } from 'react-relay/hooks';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import type { QueryRendererProps } from 'tachyon-relay';
import { TachyonQueryRenderer } from 'tachyon-relay';
import { useRouteChange } from '../../../../../routing';

type Refetch = (() => void) | null;

/**
 * Refetch the GQL request that powers the current page whenever a navigation event occurs for the same page (either by clicking a link to the current page or via `router.reload()`).
 *
 * This makes our page GraphQL requests exhibit the same behavior as documented by Next's router and `getInitialProps`: specifically, navigating to the same page (with shallow: false) will retrigger the query.
 */
export function useRefetchOnReloadCurrentPage(): MutableRefObject<Refetch> {
  const refetch = useRef<Refetch>(null);
  const { currentPathname } = useRouterUtils();
  const routeChange = useRef<{
    currentPathname: string;
    prevPathname: string | undefined;
    shallow: boolean;
  }>({ currentPathname, prevPathname: undefined, shallow: false });

  useRouteChange((path: string, { shallow }) => {
    routeChange.current = {
      currentPathname: path,
      prevPathname: routeChange.current.currentPathname,
      shallow,
    };
  });

  const change = routeChange.current;
  useEffect(() => {
    const reloaded = change.prevPathname === change.currentPathname;
    if (reloaded && !change.shallow) {
      refetch.current?.();
    }
  }, [change]);

  return refetch;
}

// istanbul ignore next: unit tests for the hook provide sufficient coverage
export const TomorrowQueryRendererRoot: FC<
  Omit<QueryRendererProps, 'environment'>
> = ({ render, ...props }) => {
  const environment = useRelayEnvironment();
  const refetch = useRefetchOnReloadCurrentPage();
  const lastRequestFailed = useRef(false);
  /**
   * When making a request after we've been in a failed state, force a network request instead of using the cache.
   * Without this, Relay will hold onto the data for the last successful fetch (because we set 'store-and-network').
   * When we're in an offline state, this will show cached data and then immediately transition to the offline page.
   *
   * This is most noticeable with the following steps:
   * 1. Navigate to the Games Directory
   * 2. Turn off your internet connection
   * 3. Navigate to Homepage
   * 4. Observe the offline experience rendered for the homepage
   * 5. Navigate back to the Games Directory
   * 6. Observe a cached online Games Directory will render, immediately followed by an offline experience for the Games Directory
   */
  const fetchPolicy = lastRequestFailed.current
    ? 'network-only'
    : 'store-and-network';

  return (
    <TachyonQueryRenderer
      {...props}
      environment={environment}
      fetchPolicy={fetchPolicy}
      render={(response) => {
        refetch.current = response.retry;
        lastRequestFailed.current = !!response.error;
        return render(response);
      }}
    />
  );
};

TomorrowQueryRendererRoot.displayName = 'TomorrowQueryRendererRoot';
