import { act, renderHook } from '@testing-library/react-hooks';
import type { RouteChangeHandler } from '../../../../../routing';
import { useRefetchOnReloadCurrentPage } from '.';

jest.mock('tachyon-next-routing-utils', () => ({
  ...jest.requireActual('tachyon-next-routing-utils'),
  useRouterUtils: () => ({ currentPathname: '/' }),
}));

let mockUseRouteChange: RouteChangeHandler;
jest.mock('../../../../../routing', () => ({
  ...jest.requireActual('../../../../../routing'),
  useRouteChange: (fn: (x: string) => void) => {
    mockUseRouteChange = fn;
  },
}));

describe(useRefetchOnReloadCurrentPage, () => {
  it('invokes the retry when present and the same route is reloaded', () => {
    const retry = jest.fn();

    // first render
    const { rerender, result } = renderHook(useRefetchOnReloadCurrentPage);
    // set refetch
    result.current.current = retry;
    // navigate
    act(() => {
      mockUseRouteChange('/', { shallow: false });
    });
    rerender();

    expect(retry).toHaveBeenCalledTimes(1);
  });

  it('does not invoke the retry when shallow route change', () => {
    const retry = jest.fn();

    // first render
    const { rerender, result } = renderHook(useRefetchOnReloadCurrentPage);
    // set refetch
    result.current.current = retry;
    // navigate
    act(() => {
      mockUseRouteChange('/', { shallow: true });
    });
    rerender();

    expect(retry).toHaveBeenCalledTimes(0);
  });

  it('invokes the retry for subsequent reloads of the same route', () => {
    const retry = jest.fn();

    // first render
    const { rerender, result } = renderHook(useRefetchOnReloadCurrentPage);
    // set refetch
    result.current.current = retry;
    // navigate
    act(() => {
      mockUseRouteChange('/', { shallow: false });
    });
    rerender();
    // navigate
    act(() => {
      mockUseRouteChange('/', { shallow: false });
    });
    rerender();
    // navigate
    act(() => {
      mockUseRouteChange('/', { shallow: false });
    });
    rerender();

    expect(retry).toHaveBeenCalledTimes(3);
  });

  it('does not invoke the retry when the route changes', () => {
    const retry = jest.fn();

    // first render
    const { rerender, result } = renderHook(useRefetchOnReloadCurrentPage);
    // set refetch
    result.current.current = retry;
    // navigate
    act(() => {
      mockUseRouteChange('/directory', { shallow: false });
    });
    rerender();

    expect(retry).toHaveBeenCalledTimes(0);
  });

  it('does not invoke the retry when revisting a previous route', () => {
    const retry = jest.fn();

    // first render
    const { rerender, result } = renderHook(useRefetchOnReloadCurrentPage);
    // set refetch
    result.current.current = retry;
    // navigate
    act(() => {
      mockUseRouteChange('/directory', { shallow: false });
    });
    rerender();
    // navigate
    act(() => {
      mockUseRouteChange('/', { shallow: false });
    });
    rerender();

    expect(retry).toHaveBeenCalledTimes(0);
  });

  it('is stable across rerenders (only refetches in response to route changes)', () => {
    const retry = jest.fn();

    // first render
    const { rerender, result } = renderHook(useRefetchOnReloadCurrentPage);
    // set refetch
    result.current.current = retry;
    rerender();
    rerender();

    expect(retry).toHaveBeenCalledTimes(0);
  });
});
