import type { Variables as RelayQueryVariables } from 'react-relay/hooks';
import { fetchQuery } from 'react-relay/hooks';
import { getCurrentUserOnServer, logoutOnServer } from 'tachyon-auth-server';
import type { TachyonPageContext, TachyonPageType } from 'tachyon-next-types';
import type { FetchQueryOpts, RecordMap } from 'tachyon-relay';
import { initEnvironment, isValidQueryResponse } from 'tachyon-relay';
import { getTachyonEnvVar } from 'tachyon-server-utils';
import { HTTPStatusCode } from 'tachyon-type-library';
import { buildCacheHeader } from 'tachyon-utils';
import type { SetRequired } from 'type-fest';
import type { TomorrowRequestExtensions } from '../../../../config';
import { DEFAULT_CACHE_VALUES } from '../../../../config';

type ServersideTachyonPageContext<
  Ctx extends TachyonPageContext = TachyonPageContext<
    {},
    TomorrowRequestExtensions,
    {}
  >,
> = Ctx & SetRequired<Ctx, 'req' | 'res'>;

export function isServersideContext<
  Ctx extends TachyonPageContext<{}, TomorrowRequestExtensions, {}>,
>(ctx: Ctx): ctx is ServersideTachyonPageContext<Ctx> {
  return ctx.req !== undefined && ctx.res !== undefined;
}

type GetServersideInitialPropsOpts<
  Ctx extends ServersideTachyonPageContext,
  Page extends TachyonPageType,
> = {
  Component: Page;
  fetchQueryOpts: FetchQueryOpts;
  relayQueryVariables: RelayQueryVariables;
  tachyonCtx: Ctx;
};

type ServersideInitialPropsError = {
  isLoggedInServerside: boolean;
  relayQueryRecords: undefined;
  serversideError: true;
};

type ServersideInitialPropsSuccess = {
  isLoggedInServerside: boolean;
  relayQueryRecords: RecordMap | undefined;
  serversideError: false;
};

export type TomorrowServersideInitialProps =
  | ServersideInitialPropsError
  | ServersideInitialPropsSuccess;

/**
 * Derives server side props by contacting GQL if necessary and sets the page's statusCode
 */
export async function getServersideInitialProps<
  Ctx extends ServersideTachyonPageContext,
  Page extends TachyonPageType,
>({
  Component,
  fetchQueryOpts,
  relayQueryVariables,
  tachyonCtx,
}: GetServersideInitialPropsOpts<
  Ctx,
  Page
>): Promise<TomorrowServersideInitialProps> {
  // we only pass deviceId for non-cached pages
  const deviceId =
    Component.totalCacheLife === 0
      ? tachyonCtx.req.tachyon.deviceId
      : undefined;

  let isGQLPartialSuccess = false;
  let relayQueryRecords;
  let relayQueryResponse;
  const currentUser = getCurrentUserOnServer(tachyonCtx.req, tachyonCtx.res);
  let isLoggedInServerside = currentUser.isLoggedIn;

  if (Component.query) {
    const relayEnvironment = initEnvironment({
      fetchQueryOpts: {
        ...fetchQueryOpts,
        // TODO: Dev / Staging only until https://jira.xarth.tv/browse/AWSI-4514
        authorization:
          getTachyonEnvVar() === 'production'
            ? undefined
            : {
                token: currentUser.authorizationToken,
                unauthorizedHandler: () => {
                  isLoggedInServerside = false;
                  logoutOnServer(tachyonCtx.res);
                },
              },
        deviceId,
        nonFatalErrorsHandler: () => {
          isGQLPartialSuccess = true;
        },
      },
    });

    try {
      relayQueryResponse = await fetchQuery(
        relayEnvironment,
        Component.query,
        relayQueryVariables,
      ).toPromise();
    } catch (error) {
      // Use a specific non-500 status code to uniquely identify errors
      // that originated from the GQL API Gateway or a backend service.
      tachyonCtx.res.statusCode = HTTPStatusCode.BadGateway;
    }

    if (isValidQueryResponse(relayQueryResponse)) {
      const componentInitialProps =
        (await Component.getInitialProps?.(tachyonCtx)) ?? {};

      const componentProps = {
        ...componentInitialProps,
        ...relayQueryResponse,
      };

      // each page is responsible for actually rendering its own 404, this
      // only controls the page's response status code
      const isNotFoundServerside =
        Component.isNotFoundServerside?.(componentProps);

      if (isNotFoundServerside) {
        tachyonCtx.res.statusCode = HTTPStatusCode.NotFound;
      }

      relayQueryRecords = relayEnvironment.getStore().getSource().toJSON();
    }
  }

  if (isLoggedInServerside) {
    tachyonCtx.res.setHeader(
      'Cache-Control',
      buildCacheHeader({ totalCacheLife: 0 }),
    );
  } else if (!Component.query || isValidQueryResponse(relayQueryResponse)) {
    tachyonCtx.res.setHeader(
      'Cache-Control',
      buildCacheHeader({ ...DEFAULT_CACHE_VALUES, ...Component }),
    );
  }

  const statusCode = tachyonCtx.res.statusCode;
  const error500 = 500 <= statusCode && statusCode <= 599;
  const serversideError = !!tachyonCtx.err || error500;

  // Set statusCode to 5XX so we don't cache or index partial success pages.
  // This does not set serversideError.
  // Not sure why, but TS infers the type to always be false here
  // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  if (isGQLPartialSuccess) {
    tachyonCtx.res.statusCode = HTTPStatusCode.BadGateway;
  }

  return serversideError
    ? {
        isLoggedInServerside,
        relayQueryRecords: undefined,
        serversideError: true,
      }
    : {
        isLoggedInServerside,
        relayQueryRecords,
        serversideError: false,
      };
}
