import { getCurrentUserOnServer, logoutOnServer } from 'tachyon-auth-server';
import type { InitEnvironmentOpts, TwitchPayloadError } from 'tachyon-relay';
import { initEnvironment } from 'tachyon-relay';
import { getTachyonEnvVar } from 'tachyon-server-utils';
import { HTTPStatusCode } from 'tachyon-type-library';
import { buildCacheHeader } from 'tachyon-utils';
import { DEFAULT_CACHE_VALUES } from '../../../../config';
import { getServersideInitialProps } from '.';

let mockFetchQueryValue: Promise<any> | undefined = undefined;
jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  fetchQuery: jest.fn(() => ({ toPromise: () => mockFetchQueryValue })),
}));

let mockNonFatalErrors = false;
jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  initEnvironment: jest.fn((opts: InitEnvironmentOpts) => {
    if (mockNonFatalErrors) {
      opts.fetchQueryOpts!.nonFatalErrorsHandler!([
        'uh oh',
      ] as unknown as TwitchPayloadError[]);
    }
    return jest.requireActual('tachyon-relay').initEnvironment(opts);
  }),
}));
const mockInitEnvironment = initEnvironment as jest.Mock;

jest.mock('tachyon-server-utils', () => ({
  getTachyonEnvVar: jest.fn(() => 'development'),
}));
const mockGetTachyonEnvVar = getTachyonEnvVar as jest.Mock;

jest.mock('tachyon-auth-server', () => ({
  getCurrentUserOnServer: jest.fn(() => ({ isLoggedIn: false })),
  logoutOnServer: jest.fn(),
}));
const mockLogoutOnServer = logoutOnServer as jest.Mock;
const mockGetCurrentUserOnServer = getCurrentUserOnServer as jest.Mock;

const mockTachyonRes = { setHeader: jest.fn() };

beforeEach(() => {
  mockFetchQueryValue = undefined;
  mockTachyonRes.setHeader.mockReset();
});

describe(getServersideInitialProps, () => {
  describe('authentication', () => {
    it('provides the authToken when present (and not production)', async () => {
      const mockToken = 'token';
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        authorizationToken: mockToken,
        isLoggedIn: true,
      }));

      await getServersideInitialProps({
        Component: { query: true } as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(mockInitEnvironment).toHaveBeenCalledWith(
        expect.objectContaining({
          fetchQueryOpts: expect.objectContaining({
            authorization: expect.objectContaining({ token: mockToken }),
          }),
        }),
      );
    });

    it('does not provide the authToken when production', async () => {
      mockGetTachyonEnvVar.mockImplementationOnce(() => 'production');
      const mockToken = 'token';
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        authorizationToken: mockToken,
        isLoggedIn: true,
      }));

      await getServersideInitialProps({
        Component: { query: true } as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(mockInitEnvironment).not.toHaveBeenCalledWith(
        expect.objectContaining({
          fetchQueryOpts: expect.objectContaining({
            authorization: expect.objectContaining({ token: mockToken }),
          }),
        }),
      );
    });

    it('logs the user out when unauthorized', async () => {
      const mockToken = 'token';
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        authorizationToken: mockToken,
        isLoggedIn: true,
      }));
      mockInitEnvironment.mockImplementationOnce((opts) => {
        opts.fetchQueryOpts.authorization.unauthorizedHandler();
      });

      const { isLoggedInServerside } = await getServersideInitialProps({
        Component: { query: true } as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(mockLogoutOnServer).toHaveBeenCalledTimes(1);
      expect(isLoggedInServerside).toEqual(false);
    });

    it('returns currentUser status', async () => {
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        isLoggedIn: true,
      }));

      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.isLoggedInServerside).toEqual(true);
    });
  });

  describe('serversideError', () => {
    it('true when tachyonCtx.err', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          err: new Error('oh no'),
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.serversideError).toEqual(true);
    });

    it('true when 500 status code', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 500 },
        } as any,
      });

      expect(res.serversideError).toEqual(true);
    });

    it('false otherwise', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.serversideError).toEqual(false);
    });
  });

  describe('relayQueryRecords', () => {
    describe('Component with query', () => {
      it('returns relay query records', async () => {
        mockFetchQueryValue = Promise.resolve(true);

        const res = await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            res: { ...mockTachyonRes, statusCode: 200 },
          } as any,
        });

        expect(res.relayQueryRecords).not.toBeUndefined();
      });
    });

    describe('Component without query', () => {
      it('does not return relay query records', async () => {
        const res = await getServersideInitialProps({
          Component: {} as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            res: { ...mockTachyonRes, statusCode: 200 },
          } as any,
        });

        expect(res.relayQueryRecords).toBeUndefined();
      });
    });
  });

  describe('headers', () => {
    const defaultCacheValues = buildCacheHeader(DEFAULT_CACHE_VALUES);

    describe('Caching', () => {
      it('sets "Cache-Control" header when valid GQL response', async () => {
        mockFetchQueryValue = Promise.resolve(true);

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res: mockTachyonRes } as any,
        });

        expect(mockTachyonRes.setHeader).toHaveBeenCalledWith(
          'Cache-Control',
          defaultCacheValues,
        );
      });

      it('does not set "Cache-Control" header when GQL request errors', async () => {
        mockFetchQueryValue = Promise.resolve(null);

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res: mockTachyonRes } as any,
        });

        expect(mockTachyonRes.setHeader).toHaveBeenCalledTimes(0);
      });
    });

    it('sets "Cache-Control" header for Component without a query', async () => {
      await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: { res: mockTachyonRes } as any,
      });

      expect(mockTachyonRes.setHeader).toHaveBeenCalledWith(
        'Cache-Control',
        defaultCacheValues,
      );
    });

    it('does not cache when the user is logged in', async () => {
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        authorizationToken: 'authToken',
        isLoggedIn: true,
      }));

      await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: { res: mockTachyonRes } as any,
      });

      expect(mockTachyonRes.setHeader).toHaveBeenCalledWith(
        'Cache-Control',
        buildCacheHeader({
          totalCacheLife: 0,
        }),
      );
    });
  });

  describe('Component with query', () => {
    it('request failures set the status code to BadGateway', async () => {
      const error = new Error('oh no');
      mockFetchQueryValue = Promise.reject(error);
      const res = { ...mockTachyonRes, statusCode: undefined };

      await getServersideInitialProps({
        Component: { query: true } as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: { res } as any,
      });

      expect(res.statusCode).toEqual(HTTPStatusCode.BadGateway);
    });

    it('partial request failures set the status code to BadGateway', async () => {
      mockNonFatalErrors = true;
      const res = { ...mockTachyonRes, statusCode: undefined };

      await getServersideInitialProps({
        Component: { query: true } as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: { res } as any,
      });

      expect(res.statusCode).toEqual(HTTPStatusCode.BadGateway);
      mockNonFatalErrors = false;
    });

    describe('Component.isNotFoundServerside', () => {
      it('sets the status code to NotFound', async () => {
        mockFetchQueryValue = Promise.resolve(true);
        const res = { ...mockTachyonRes, statusCode: undefined };

        await getServersideInitialProps({
          Component: { isNotFoundServerside: () => true, query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res } as any,
        });

        expect(res.statusCode).toEqual(HTTPStatusCode.NotFound);
      });
    });
  });
});
