import App from 'next/app';
import Head from 'next/head';
import type { FC } from 'react';
import type { Variables as RelayQueryVariables } from 'react-relay/hooks';
import { CurrentUserRoot } from 'tachyon-auth';
import type { DynamicSettings } from 'tachyon-dynamic-settings';
import { DynamicSettingsRoot } from 'tachyon-dynamic-settings';
import type { AppEnvironment } from 'tachyon-environment';
import { TachyonIntlRoot } from 'tachyon-intl';
import type { ClientSideErrorBoundaryProps } from 'tachyon-more-ui';
import { ClientSideErrorBoundary } from 'tachyon-more-ui';
import { RouterUtilsRoot } from 'tachyon-next-routing-utils';
import type {
  TachyonAppBaseInitialProps,
  TachyonAppBaseProps,
  TachyonAppFC,
  TachyonPageInitialProps,
} from 'tachyon-next-types';
import {
  CHANNEL_VODS_FILTER_QUERY_PARAM_KEY,
  useErrorPath,
} from 'tachyon-page-utils';
import type { FetchQueryOpts, QueryRendererRenderProps } from 'tachyon-relay';
import { RequestInfoRoot } from 'tachyon-relay';
import { isBrowser } from 'tachyon-utils';
import {
  APP_SHELL_REDIRECT_QUERY_KEY,
  CHANNEL_CLIPS_FILTER_QUERY_PARAM_KEY,
  EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY,
  PLAYER_DEBUG_MODE_QUERY_PARAM_KEY,
  PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY,
  SEARCH_TERM_QUERY_PARAM_KEY,
  SEARCH_TYPE_QUERY_PARAM_KEY,
  TAG_QUERY_PARAM_KEY,
} from '../../../config';
import type {
  TomorrowDynamicSettings,
  TomorrowPageExtensions,
  TomorrowRequestExtensions,
} from '../../../config';
import { ThemeRoot } from '../../../theme';
import {
  AddToHomeScreenCTA,
  AddToHomeScreenRoot,
  CookieNotice,
} from '../../common';
import { TomorrowBranchRoot } from '../../growth';
import { MinimalInternalError } from './MinimalInternalError';
import { PageRenderer } from './PageRenderer';
import { ServiceWorker } from './ServiceWorker';
import { configure } from './configure';
import {
  TomorrowDebugRoot,
  TomorrowDiscoveryTrackingRoot,
  TomorrowEnvironmentRoot,
  TomorrowEventReporterRoot,
  TomorrowEventTrackerRoot,
  TomorrowExperimentsRoot,
  TomorrowLatencyTrackerRoot,
  TomorrowQueryRendererRoot,
  TomorrowRelayEnvironmentRoot,
} from './context-root-adapters';
import type { TomorrowServersideInitialProps } from './getServersideInitialProps';
import {
  getServersideInitialProps,
  isServersideContext,
} from './getServersideInitialProps';

configure();

// list of query params to protect from being stripped by RouterUtilsRoot
// (allows them to persist through reloads)
const PRESERVED_QUERY_PARAMS = [
  // Allow QA to override the assigned bucket
  EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY,
  // Allow for player-core debugging
  PLAYER_DEBUG_MODE_QUERY_PARAM_KEY,
  // Index into a VOD at a specific time point
  PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY,
  // Allow search queries to be provided to search page
  SEARCH_TERM_QUERY_PARAM_KEY,
  // Allow search category filtering to be provided to search page
  SEARCH_TYPE_QUERY_PARAM_KEY,
  // Allow tag filtering on streams
  TAG_QUERY_PARAM_KEY,
  // Allow list filtering on channel clips page
  CHANNEL_CLIPS_FILTER_QUERY_PARAM_KEY,
  // Allow list filtering on channel videos page
  CHANNEL_VODS_FILTER_QUERY_PARAM_KEY,
  // AppShell cleans this up itself, preserving this prevents any races
  APP_SHELL_REDIRECT_QUERY_KEY,
];

// non-localized error UI of last resort
const RootErrorMessage: FC = () => (
  <span>An error occurred. Please refresh the page.</span>
);
RootErrorMessage.displayName = 'RootErrorMessage';

/**
 * Values that we need to transport from the server to client, and we lean on
 * Next's `getInitialProps` serialization functionality to achieve that. The
 * values are set on the server, consumed from __NEXT_DATA__ via normal Next
 * functionality for the first page render, and then manually re-consumed
 * on subsequent app re-renders to ensure their continued availability.
 */
type PersistedTomorrowAppInitialProps = Pick<
  TomorrowRequestExtensions,
  'experimentBucket' | 'initialTheme' | 'intlData'
> & {
  appEnvironment: AppEnvironment;
  dynamicSettings: DynamicSettings<TomorrowDynamicSettings>;
  language: string;
};

// Be very conservative about what is put in here, since all of this will get
// serialized for being sent to the client and big/complex objects both don't
// do very well with this and also cause a ton of page bloat.
type TomorrowAppInitialProps = Partial<TomorrowServersideInitialProps> &
  PersistedTomorrowAppInitialProps &
  TachyonAppBaseInitialProps & {
    currentAsPath: string;
    fetchQueryOpts: FetchQueryOpts;
    relayQueryVariables: RelayQueryVariables;
  };

export type TomorrowAppProps = TachyonAppBaseProps & TomorrowAppInitialProps;

export const TomorrowApp: TachyonAppFC<
  TomorrowAppInitialProps,
  TomorrowAppProps,
  TomorrowRequestExtensions,
  {},
  TomorrowPageExtensions<{}>
> = ({
  Component,
  appEnvironment,
  currentAsPath,
  dynamicSettings,
  experimentBucket,
  fetchQueryOpts,
  initialTheme,
  intlData,
  isLoggedInServerside,
  language,
  pageProps,
  relayQueryRecords,
  relayQueryVariables,
  serversideError,
}) => {
  const [erroringPath, onError] = useErrorPath(currentAsPath);
  const inError = serversideError ?? !!erroringPath;

  const commonErrorBoundaryProps: Pick<
    ClientSideErrorBoundaryProps,
    'app' | 'currentPath' | 'isDevelopment'
  > = {
    app: 'tomorrow',
    currentPath: currentAsPath,
    isDevelopment: process.env.NODE_ENV === 'development',
  };

  return (
    <ClientSideErrorBoundary
      {...commonErrorBoundaryProps}
      boundaryName="TomorrowRootBoundary"
      errorStateComponent={RootErrorMessage}
      isRootBoundary
    >
      <Head>
        <meta
          // the viewport meta tag can't be placed in _document: https://github.com/vercel/next.js/blob/master/errors/no-document-viewport-meta.md
          content="width=device-width, initial-scale=1, viewport-fit=cover"
          name="viewport"
        />
      </Head>
      <RouterUtilsRoot preservedParams={PRESERVED_QUERY_PARAMS}>
        <DynamicSettingsRoot settings={dynamicSettings}>
          <TomorrowEnvironmentRoot
            appEnvironment={appEnvironment}
            language={language}
          >
            <TomorrowEventReporterRoot>
              <TomorrowDebugRoot />
              <TachyonIntlRoot data={intlData}>
                <CurrentUserRoot isLoggedInServerside={isLoggedInServerside}>
                  <TomorrowEventTrackerRoot inError={inError}>
                    <TomorrowRelayEnvironmentRoot
                      fetchQueryOpts={fetchQueryOpts}
                      records={relayQueryRecords}
                    >
                      <RequestInfoRoot>
                        <TomorrowLatencyTrackerRoot>
                          <TomorrowExperimentsRoot
                            bucket={experimentBucket}
                            debug={appEnvironment === 'development'}
                            experiments={dynamicSettings.experiments}
                          >
                            <TomorrowDiscoveryTrackingRoot>
                              <ClientSideErrorBoundary
                                {...commonErrorBoundaryProps}
                                boundaryName="TomorrowAppBoundary"
                                errorStateComponent={MinimalInternalError}
                                onError={onError}
                              >
                                <ThemeRoot initialTheme={initialTheme}>
                                  {serversideError ? (
                                    <MinimalInternalError />
                                  ) : (
                                    <AddToHomeScreenRoot>
                                      <TomorrowBranchRoot>
                                        <TomorrowQueryRendererRoot
                                          query={Component.query}
                                          render={({
                                            error,
                                            props: relayQueryResponse,
                                          }: QueryRendererRenderProps) => {
                                            return (
                                              <PageRenderer
                                                Page={Component}
                                                error={error}
                                                pageProps={pageProps}
                                                relayQueryResponse={
                                                  relayQueryResponse
                                                }
                                              />
                                            );
                                          }}
                                          variables={relayQueryVariables}
                                        />
                                      </TomorrowBranchRoot>
                                      <CookieNotice />
                                      <AddToHomeScreenCTA />
                                    </AddToHomeScreenRoot>
                                  )}
                                </ThemeRoot>
                                <ServiceWorker />
                              </ClientSideErrorBoundary>
                            </TomorrowDiscoveryTrackingRoot>
                          </TomorrowExperimentsRoot>
                        </TomorrowLatencyTrackerRoot>
                      </RequestInfoRoot>
                    </TomorrowRelayEnvironmentRoot>
                  </TomorrowEventTrackerRoot>
                </CurrentUserRoot>
              </TachyonIntlRoot>
            </TomorrowEventReporterRoot>
          </TomorrowEnvironmentRoot>
        </DynamicSettingsRoot>
      </RouterUtilsRoot>
    </ClientSideErrorBoundary>
  );
};

TomorrowApp.getInitialProps = async (appContext) => {
  const { Component, ctx: tachyonCtx } = appContext;

  const persistedProps: PersistedTomorrowAppInitialProps = isBrowser()
    ? {
        /* eslint-disable @typescript-eslint/no-unsafe-assignment */
        appEnvironment: window.__NEXT_DATA__.props.appEnvironment,
        dynamicSettings: window.__NEXT_DATA__.props.dynamicSettings,
        experimentBucket: window.__NEXT_DATA__.props.experimentBucket,
        initialTheme: window.__NEXT_DATA__.props.initialTheme,
        intlData: window.__NEXT_DATA__.props.intlData,
        language: window.__NEXT_DATA__.props.language,
        /* eslint-enable @typescript-eslint/no-unsafe-assignment */
      }
    : {
        /* eslint-disable @typescript-eslint/no-non-null-assertion */
        appEnvironment: tachyonCtx.req!.tachyon.appEnvironment,
        dynamicSettings: tachyonCtx.req!.tachyon.dynamicSettings,
        experimentBucket: tachyonCtx.req!.tachyon.experimentBucket,
        initialTheme: tachyonCtx.req!.tachyon.initialTheme,
        intlData: tachyonCtx.req!.tachyon.intlData,
        language: tachyonCtx.req!.tachyon.intlData.preferredLanguageTags[0],
        /* eslint-enable @typescript-eslint/no-non-null-assertion */
      };

  // update context before calling super.getInitialProps to make values
  // available to current page component
  tachyonCtx.language = persistedProps.language;
  const initialProps = await App.getInitialProps(appContext);

  const fetchQueryOpts: FetchQueryOpts = {
    language: persistedProps.language,
  };

  const relayQueryVariables: RelayQueryVariables =
    (initialProps.pageProps as TachyonPageInitialProps).queryVariables ?? {};

  const serverSideInitialProps =
    // typeof window and tachyonCtx.req/res should move in sync, here just for types
    typeof window === 'undefined' && isServersideContext(tachyonCtx)
      ? await getServersideInitialProps({
          Component,
          fetchQueryOpts,
          relayQueryVariables,
          tachyonCtx,
        })
      : undefined;

  return {
    ...initialProps,
    ...persistedProps,
    ...serverSideInitialProps,
    currentAsPath: tachyonCtx.asPath!, // eslint-disable-line @typescript-eslint/no-non-null-assertion
    fetchQueryOpts,
    relayQueryVariables,
  };
};

TomorrowApp.displayName = 'TomorrowApp';
