import { lorem } from 'faker';
import { mockIntlData } from 'tachyon-intl';
import { RouterUtilsRoot } from 'tachyon-next-routing-utils';
import type {
  TachyonPageInitialProps,
  TachyonPageProps,
  TachyonPageType,
} from 'tachyon-next-types';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { isBrowser } from 'tachyon-utils';
import type { TomorrowPageExtensions } from '../../../config';
import { MinimalInternalError } from './MinimalInternalError';
import { TomorrowQueryRendererRoot } from './context-root-adapters';
import { getServersideInitialProps } from './getServersideInitialProps';
import type { TomorrowAppProps } from '.';
import { TomorrowApp } from '.';

/* supress console messages when running tests */
const mockConsoleDebug = jest.spyOn(console, 'debug').mockImplementation();
const mockConsoleError = jest.spyOn(console, 'error').mockImplementation();
const mockConsoleWarn = jest.spyOn(console, 'warn').mockImplementation();
afterAll(() => {
  mockConsoleDebug.mockRestore();
  mockConsoleError.mockRestore();
  mockConsoleWarn.mockRestore();
});

jest.mock('./ServiceWorker', () => ({
  ServiceWorker: () => <div />,
}));

jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  RequestInfoRoot: ({ children }: any) => children,
}));

jest.mock('next/router', () => ({
  useRouter: () => ({ events: { off: jest.fn(), on: jest.fn() } }),
}));

jest.mock('tachyon-next-routing-utils', () => {
  return {
    ...jest.requireActual('tachyon-next-routing-utils'),
    RouterUtilsRoot: jest.fn((props: any) => props.children),
  };
});
const mockRouterUtilsRoot = RouterUtilsRoot as jest.Mock;

jest.mock('./configure', () => ({
  configure: jest.fn(),
}));

jest.mock('next/app', () => {
  const app = () => false;
  app.getInitialProps = () => ({ pageProps: { queryVariables: {} } });
  return {
    __esModule: true,
    default: app,
  };
});

jest.mock('tachyon-debug-reporter', () => ({
  configTachyonDebug: jest.fn(),
}));

jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  isBrowser: jest.fn(() => true),
}));
const mockIsBrowser = isBrowser as jest.Mock;

const mockGetServersideInitialProps = getServersideInitialProps as jest.Mock;
jest.mock('./getServersideInitialProps', () => ({
  ...jest.requireActual('./getServersideInitialProps'),
  getServersideInitialProps: jest.fn(() => ({ isLoggedInServerside: false })),
}));

const mockPage = (
  renderer: any = () => <div />,
): TachyonPageType<
  TachyonPageInitialProps,
  TachyonPageProps,
  TomorrowPageExtensions<{}>
> => {
  const page = renderer;
  page.pageviewTracking = defaultPageviewTracking;

  return page;
};

describe(TomorrowApp, () => {
  const pageProps = { foo: 'bar' };
  const intlData = mockIntlData();

  const setup = createMountWrapperFactory(
    TomorrowApp as any,
    (): TomorrowAppProps => ({
      Component: mockPage(),
      appEnvironment: 'development',
      currentAsPath: '/',
      dynamicSettings: {
        cookieNoticeCountryCodes: [],
        cookiePolicyUrl: '',
        environment: 'development',
        experiments: {},
        flumeAllowedEvents: [],
        flumeAllowedProperties: [],
        privacyPolicyUrl: '',
        spadeUrl: '//spade',
        termsOfServiceUrl: '',
      },
      experimentBucket: 'a',
      fetchQueryOpts: {},
      initialTheme: lorem.word(),
      intlData,
      language: intlData.preferredLanguageTags[0],
      pageProps: pageProps as any,
      relayQueryVariables: {},
      router: {} as any,
      serversideError: false,
    }),
  );

  beforeEach(() => {
    // @ts-expect-error: tests
    window.matchMedia = jest.fn(() => ({ matches: true }));
  });

  afterEach(() => {
    // @ts-expect-error: tests
    delete window.matchMedia;
  });

  it('renders <MinimalInternalError /> when there is an SSR error', () => {
    const { wrapper } = setup({ serversideError: true });

    expect(wrapper.find(MinimalInternalError)).toExist();
  });

  it('renders the TomorrowAppBoundary when the <Page/> fails', () => {
    const { wrapper } = setup({
      Component: mockPage(() => {
        throw new Error('oh no');
      }),
    });

    expect(wrapper.find({ boundaryName: 'TomorrowAppBoundary' })).toExist();
    expect(wrapper.find(MinimalInternalError)).toExist();
  });

  it('renders the TomorrowRootBoundary when the app fails', () => {
    mockRouterUtilsRoot.mockImplementationOnce(() => {
      throw new Error('oh no');
    });
    const { wrapper } = setup();

    expect(wrapper.find({ boundaryName: 'TomorrowRootBoundary' })).toExist();
    expect(wrapper).toIncludeText(
      'An error occurred. Please refresh the page.',
    );
  });

  it('renders the <TomorrowQueryRendererRoot /> when there is not an SSR error', () => {
    const { wrapper } = setup();

    expect(wrapper.find(TomorrowQueryRendererRoot)).toExist();
  });
});

describe('TomorrowApp.getInitialProps', () => {
  describe('client side', () => {
    const intlData = mockIntlData();

    const mockNextData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
      experimentBucket: lorem.word(),
      initialTheme: lorem.word(),
      intlData,
      language: intlData.preferredLanguageTags[0],
    };

    beforeEach(() => {
      global.window.__NEXT_DATA__ = { props: mockNextData } as any;
    });

    // @ts-expect-error: tests
    afterEach(() => delete global.window.__NEXT_DATA__);

    it('extracts __NEXT_DATA__ fields', async () => {
      const ip = await TomorrowApp.getInitialProps({ ctx: {} } as any);

      expect(ip).toEqual(expect.objectContaining(mockNextData));
    });
  });

  describe('server side', () => {
    let mockWindow: any;
    beforeEach(() => {
      mockIsBrowser.mockImplementationOnce(() => false);
      mockWindow = jest.spyOn(global as any, 'window', 'get');
      mockWindow.mockImplementation(() => undefined);
    });
    afterEach(() => {
      mockWindow.mockRestore();
    });

    const mockReqData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
      experimentBucket: lorem.word(),
      initialTheme: lorem.word(),
      intlData: mockIntlData(),
    };

    const mockCtx = {
      ctx: { req: { tachyon: mockReqData }, res: {} },
    };

    it('extracts req fields when', async () => {
      const ip = await TomorrowApp.getInitialProps(mockCtx as any);

      expect(ip).toEqual(
        expect.objectContaining({
          ...mockReqData,
          language: mockReqData.intlData.preferredLanguageTags[0],
        }),
      );
    });

    it('invokes getServersideInitialProps', async () => {
      await TomorrowApp.getInitialProps(mockCtx as any);

      expect(mockGetServersideInitialProps).toHaveBeenCalledTimes(1);
    });
  });
});
