import type { DocumentInitialProps, DocumentProps } from 'next/document';
import Document, { Head, Html, Main, NextScript } from 'next/document';
import { PulsarCorePreloadWASM } from 'pulsar';
import { ServerStyleSheet } from 'styled-components';
import { PerformancePolyfill } from 'tachyon-latency-tracker';
// TODO: Consider enabling Core UI Fonts https://jira.xarth.tv/browse/EMP-4579
// import { Fonts } from 'tachyon-more-ui';
import type { TachyonDocumentContext } from 'tachyon-next-types';
import { PolyfillPreload, PolyfillScript } from 'tachyon-polyfill';
import type { PolyfillProps } from 'tachyon-polyfill';
import { GraphqlPreconnect } from 'tachyon-relay';
import type { TomorrowRequestExtensions } from '../../../config';

export type TomorrowDocumentInitialProps = DocumentInitialProps & {
  currentPathname: string;
  language: string;
};

export type TomorrowDocumentProps = DocumentProps &
  TomorrowDocumentInitialProps;

// This component is beginning to do webpack imports in order to enable plugins,
// which is really messing with Jest
/* istanbul ignore next: mostly mocks */
/**
 * Root-level React component that is only rendered server-side, see:
 * https://nextjs.org/docs/#custom-document
 */
export class TomorrowDocument extends Document<TomorrowDocumentProps> {
  public static async getInitialProps(
    context: TachyonDocumentContext<TomorrowRequestExtensions>,
  ): Promise<TomorrowDocumentInitialProps> {
    const sheet = new ServerStyleSheet();
    const originalRenderPage = context.renderPage;

    try {
      context.renderPage = () =>
        originalRenderPage({
          enhanceApp: (App) => (props) =>
            sheet.collectStyles(<App {...props} />),
        });

      const initialProps = await super.getInitialProps(context);
      return {
        ...initialProps,
        currentPathname: context.pathname,
        language: context.req!.tachyon.intlData.preferredLanguageTags[0], // eslint-disable-line @typescript-eslint/no-non-null-assertion
        styles: sheet.getStyleElement(),
      };
    } finally {
      sheet.seal();
    }
  }

  public render(): JSX.Element {
    const { language } = this.props;
    const polyfillProps: PolyfillProps = {
      features: ['smoothscroll'],
      locale: language,
    };

    return (
      <Html lang={language}>
        <Head>
          <PerformancePolyfill />
          <link href="https://static-cdn.jtvnw.net" rel="preconnect" />
          <GraphqlPreconnect />
          <PulsarCorePreloadWASM />
          <PolyfillPreload {...polyfillProps} />
          {/* TODO: Consider enabling Core UI Fonts https://jira.xarth.tv/browse/EMP-4579 <Fonts /> */}
          <link href="/manifest.json" rel="manifest" />
          <meta content="yes" name="apple-mobile-web-app-capable" />
          {/* Safari icons (Safari does not pull icons from the manifest) */}
          <link
            href="/static/images/pwa/icons/pwaicon-120.png"
            rel="apple-touch-icon"
            sizes="120x120"
          />
          <link
            href="/static/images/pwa/icons/pwaicon-152.png"
            rel="apple-touch-icon"
            sizes="152x152"
          />
          <link
            href="/static/images/pwa/icons/pwaicon-167.png"
            rel="apple-touch-icon"
            sizes="167x167"
          />
          <link
            href="/static/images/pwa/icons/pwaicon-180.png"
            rel="apple-touch-icon"
            sizes="180x180"
          />
        </Head>
        <body>
          <Main />
          <PolyfillScript {...polyfillProps} />
          <NextScript />
        </body>
      </Html>
    );
  }
}
