import type {
  ChannelAliasMiddlewareOpts,
  PageAlias,
  PageAliasMiddlewareOpts,
  RedirectToDesktopWebMiddlewareOpts,
} from 'tachyon-server-utils';

/**
 * Query param that tells the /app-shell route to redirect to another
 * destination. The value must be a full path and URL encoded. For example,
 * `%2Fdirectory` would redirect to the GamesDirectory page.
 */
export const APP_SHELL_REDIRECT_QUERY_KEY = 'destination';

/**
 * Mapping of channel aliases to their target channel.
 *
 * This is useful for custom-aliasing channels that exist but aren't intended
 * to be accessible.
 */
const channelAliases: Record<string, string> = {
  esa: 'esamarathon',
  overwatchleague_allaccess: 'overwatchleague',
};

export const CHANNEL_ALIAS_MIDDLEWARE_OPTS: ChannelAliasMiddlewareOpts = {
  channelAliases,
};

// NOTE: tune these lists via `pageAliasManager` test suite

/**
 * List of paths that should go to the channel page.
 */
// istanbul ignore next: trivial
const channelPageAlias: PageAlias = {
  aliases: [
    /^\/(?<login>\w+)\/mobile\/?$/i,
    /^\/(?<login>\w+)\/rooms\/\w+\/?$/i,
    /^\/(?<login>\w+)\/squad\/?$/i,
  ],
  pathBuilder: ({ login }) => `/${login}`,
};

/**
 * List of paths that should go to the channel home page.
 */
// istanbul ignore next: trivial
const channelHomePageAlias: PageAlias = {
  aliases: [
    /^\/(?<login>\w+)\/collections?(?:\/\w+)?\/?$/i,
    /^\/(?<login>\w+)\/events\/?$/i,
    /^\/(?<login>\w+)\/followers\/?$/i,
    /^\/(?<login>\w+)\/following\/?$/i,
    /^\/(?<login>\w+)\/profile\/?$/i,
    /^\/(?<login>\w+)\/profile\/\w+\/?$/i,
    /^\/(?<login>\w+)\/videos\/all\/?$/i,
    // old video format urls
    /^\/(?<login>\w+)\/(?:a|b|c|streams)\/\w+\/?$/i,
  ],
  pathBuilder: ({ login }) => `/${login}/home`,
};

/**
 * List of paths that should go to the vod page.
 */
const vodPageAlias: PageAlias = {
  aliases: [
    /^\/\w+\/video\/(?<videoId>\w+)\/?/i,
    /^\/videos\/(?<videoId>\w+)\/\w+\/?/i,
    /^\/video\/(?<videoId>\w+)\/?/i,
  ],
  pathBuilder: ({ videoId }) => `/videos/${videoId}`,
};

/**
 * List of paths that should go to the not-found page.
 */
const notFoundPageAlias: PageAlias = {
  aliases: [/^\/events\/\w+\/?/i],
  pathBuilder: () => `/not-found`,
};

export const PAGE_ALIAS_MIDDLEWARE_OPTS: PageAliasMiddlewareOpts = {
  pageAliases: [
    channelPageAlias,
    channelHomePageAlias,
    vodPageAlias,
    notFoundPageAlias,
  ],
};

/**
 * Query parameters that should be set on all URLs directing users to desktop.
 */
const baseRedirectQueryParameters = {
  /**
   * When the CDN receives a URL with this query param as "true" it won't
   * try to redirect the user to mobile web if they are on the mobile device.
   * Allows us to avoid an infinite redirection loop if we need to redirect
   * the user to Desktop (twitch.tv).
   */
  'no-mobile-redirect': 'true',
};

/**
 * The depth as which paths should be redirected to desktop web
 */
const pathDepthCutoff = 4;

/**
 * Paths that should always be sent back to www.
 *
 * These paths should always trigger a redirect to www, regardless of the way
 * they got to m.twitch.tv.
 */
const pathsToForceRedirect = [
  '*/chat',
  '*/dashboard',
  '*/dashboard/**',
  '*/manager',
  '*/manager/**',
  '*/subscribe',
  'about/**',
  'activate',
  'apps/**',
  'beta/**',
  'bits',
  'bits/config.json',
  'bits-checkout',
  'bits-checkout/**',
  'blizzheroes/esports/hgc-2017',
  'blog',
  'blog/**',
  'broadcast',
  'broadcast/soundtrack',
  'broadcast/studio',
  'contact',
  'creatorcamp',
  'creatorcamp/**',
  'dashboard',
  'dashboard/**',
  'directory/all/tags',
  'downloads',
  'email-verification/**',
  'event/**',
  'ext/**',
  'favicon',
  'favicon.ico',
  'friends',
  'gear-on-amazon',
  'gear-on-amazon/**',
  'giftcard',
  'inbox',
  'inventory',
  'jobs',
  'jobs/**',
  'legal',
  'legal/**',
  // login/logout is only here, not in the CDN, since it's coming soon
  'login',
  'logins/new',
  'logout',
  'manager',
  'messages/**',
  'moderator',
  'moderator/**',
  'outbox',
  'p',
  'p/**',
  'partner/**',
  'passport/callback',
  'passport-callback',
  'payments',
  // popout is only here, not in the CDN, to enable starshot flexibility
  'popout/**',
  'prime',
  'prime/**',
  'products/**',
  'pubg',
  'r/**',
  'redeem',
  'settings',
  'settings/**',
  'signup',
  'store',
  'store/games',
  'store/merch',
  'stream/**',
  'subs',
  'subs/**',
  'subscribe/**',
  'subscriptions',
  'support',
  'team/**',
  'turbo',
  'unsubscribe/**',
  'upload',
  'user/**',
  'view/**',
  'wallet',
  'watch/**',
];

/**
 * Query parameters that should be forwarded for all requests redirected to
 * desktop when they are present.
 */
export const redirectPreservedQueryParameters = [
  'device-code',
  'tt_content',
  'tt_email_id',
  'tt_medium',
];

/**
 * The URL query parameter that will be attached to a URL in order to indicate
 * that the request was redirected by the CDN from Desktop to Mobile Web.
 */
const wasRedirectedFromDesktopQueryParam = 'desktop-redirect';

export const REDIRECT_TO_DESKTOP_WEB_MIDDLEWARE_OPTS: RedirectToDesktopWebMiddlewareOpts =
  {
    baseRedirectQueryParameters,
    pathDepthCutoff,
    pathsToForceRedirect,
    redirectPreservedQueryParameters,
    wasRedirectedFromDesktopQueryParam,
  };
